<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/api_helpers.php';
require_once '../helpers/permission_modal.php';
require_once '../includes/sms_functions.php';
require_once '../includes/email_functions.php';
require_once '../includes/whatsapp_functions.php';
require_once '../includes/functions.php';

$activePage = 'credit';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Check if user has any credit-related permissions
$creditPermissions = [
    'manage_credit', 'process_credit_payment', 'waive_credit', 'send_reminder', 
    'add_credit_item', 'allow_credit_sale', 'view_customers'
];

if (!hasAnyPermission($creditPermissions)) {
    $_SESSION['error'] = "You don't have permission to access credit management.";
    header('Location: dashboard.php');
    exit;
}

// Get all settings grouped by category
$stmt = $pdo->query("SELECT * FROM store_settings ORDER BY setting_group, setting_key");
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Get WhatsApp settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'whatsapp_settings'");
$stmt->execute();
$whatsapp_settings = [];
while ($row = $stmt->fetch()) {
    $whatsapp_settings[$row['setting_key']] = $row['setting_value'];
}

// Get SMS settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'sms_settings'");
$stmt->execute();
$sms_settings = [];
while ($row = $stmt->fetch()) {
    $sms_settings[$row['setting_key']] = $row['setting_value'];
}

// Merge all settings for easy access
$settings = array_merge($settings, $whatsapp_settings, $sms_settings);

// Configuration: Show paid invoices in payment messages (set to false to hide them)
// Set to true to show paid invoices in SMS, WhatsApp, and Email messages
// Set to false to hide paid invoices and keep only payment summary information
$showPaidInvoicesInMessages = false; // Set to true to show paid invoices, false to hide them

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';

// Get store information
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email', 'website')");
$stmt->execute();
$store_settings = [];
while ($row = $stmt->fetch()) {
    $store_settings[$row['setting_key']] = $row['setting_value'];
}

$storeName = $store_settings['store_name'] ?? 'Store Name';
$storeAddress = $store_settings['store_address'] ?? 'Store Address';
$storePhone = $store_settings['store_phone'] ?? 'Store Phone';
$storeEmail = $store_settings['store_email'] ?? 'Store Email';
$storeWebsite = $store_settings['website'] ?? 'Store Website';

// Retrieve store logo data
$logoData = null;
$logoName = '';
$stmt = $pdo->prepare("SELECT setting_value, setting_binary FROM store_settings WHERE setting_key = 'store_logo' AND setting_group = 'store_info'");
$stmt->execute();
$logo = $stmt->fetch();
if ($logo && $logo['setting_binary']) {
    $logoData = $logo['setting_binary'];
    $logoName = $logo['setting_value'];
}

// Get credit customers summary using improved unpaid credit calculation (including POS payments)
$stmt = $pdo->prepare("
    WITH all_credit_balances AS (
        SELECT 
            ct.customer_id,
            ct.amount,
            -- Credit payments from credit_payments table (only actual payments toward credit debt, excluding refunds)
            COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) as paid,
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
            -- Subtract refunds from the original amount
            COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as refunded,
            ct.amount - COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as balance
        FROM credit_transactions ct
        LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
        GROUP BY ct.id, ct.customer_id, ct.amount
    ),
    unpaid_balances AS (
        SELECT * FROM all_credit_balances WHERE balance > 0
    )
    SELECT 
        c.id,
        c.name,
        COALESCE(SUM(acb.amount - acb.refunded), 0) as total_credit,
        COALESCE(SUM(acb.paid), 0) as total_paid,
        COALESCE(SUM(acb.waived), 0) as total_waived,
        COALESCE(SUM(ub.balance), 0) as total_balance
    FROM customers c
    LEFT JOIN all_credit_balances acb ON c.id = acb.customer_id
    LEFT JOIN unpaid_balances ub ON c.id = ub.customer_id
    WHERE acb.customer_id IS NOT NULL
    GROUP BY c.id, c.name
    HAVING total_balance > 0
    ORDER BY total_balance DESC
");
$stmt->execute();
$creditCustomers = $stmt->fetchAll();

// Get recent credit transactions using improved unpaid credit calculation (including POS payments)
$stmt = $pdo->prepare("
    SELECT 
        ct.*,
        s.invoice_number as invoice_no,
        s.created_at as date,
        c.name as customer_name,
        -- Credit payments only (actual payments toward credit debt, excluding refunds)
        COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) as paid,
        COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
        -- Subtract refunds from the original amount
        COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as refunded,
        ct.amount - COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) - 
        COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) - 
        COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as balance,
        CASE 
            WHEN (
                COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) +
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) +
                COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0)
            ) >= ct.amount THEN 'Paid'
            WHEN (
                COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) +
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) +
                COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0)
            ) > 0 THEN 'Partially Paid'
            ELSE 'Pending'
        END as status
    FROM credit_transactions ct
    JOIN sales s ON ct.sale_id = s.id
    JOIN customers c ON ct.customer_id = c.id
    LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
    GROUP BY ct.id
    ORDER BY s.created_at DESC
    LIMIT 20
");
$stmt->execute();
$recentTransactions = $stmt->fetchAll();

// Calculate total credit amounts
$totalCredit = array_sum(array_column($creditCustomers, 'total_balance'));
$amountPaid = array_sum(array_column($creditCustomers, 'total_paid'));
$pendingAmount = array_sum(array_column($creditCustomers, 'total_balance'));

// Handle payment action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && (isset($_POST['payment_id']) || isset($_POST['payment_customer_id']))) {
    header('Content-Type: application/json');
    
    try {
    $userId = $_SESSION['user_id'];
    $paymentMethod = $_POST['payment_method'];
    $reference = $_POST['reference'] ?? '';
    $amount = floatval($_POST['amount']);
    $selectedInvoiceIds = isset($_POST['payment_invoice_ids']) ? array_map('intval', (array)$_POST['payment_invoice_ids']) : [];
        
        // Log the incoming request
        error_log("Payment request received - Amount: $amount, Method: $paymentMethod, Customer ID: " . ($_POST['payment_customer_id'] ?? 'N/A'));
        
    if (empty($selectedInvoiceIds)) {
            throw new Exception("Please select at least one unpaid invoice to pay.");
        }

        if (!isset($_POST['payment_customer_id']) || empty($_POST['payment_customer_id'])) {
            throw new Exception("Customer ID is required.");
        }

        $customerId = $_POST['payment_customer_id'];
        $pdo->beginTransaction();
        
        // Get customer details
        $stmt = $pdo->prepare("SELECT name FROM customers WHERE id = ?");
        $stmt->execute([$customerId]);
        $customerName = $stmt->fetchColumn();
        
        if (!$customerName) {
            throw new Exception("Customer not found");
        }
        
        // Get selected invoices with their balances
        $inClause = implode(',', array_fill(0, count($selectedInvoiceIds), '?'));
        $params = $selectedInvoiceIds;
        $sql = "
            SELECT 
                ct.id,
                ct.sale_id,
                ct.amount,
                   s.invoice_number,
                (SELECT COALESCE(SUM(amount), 0) FROM credit_payments WHERE credit_transaction_id = ct.id AND (is_waived = 0 OR is_waived IS NULL) AND payment_method != 'refund') as paid_amount,
                   GROUP_CONCAT(
                       CONCAT(p.name, 
                           CASE 
                               WHEN psn.serial_numbers IS NOT NULL AND psn.serial_numbers != '' 
                               THEN CONCAT(' [SN: ', psn.serial_numbers, ']') 
                               ELSE '' 
                           END
                       ) SEPARATOR ', '
                   ) as items,
                COALESCE(SUM(si.quantity), 0) as total_quantity
            FROM credit_transactions ct
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN sale_items si ON s.id = si.sale_id
            LEFT JOIN products p ON si.product_id = p.id
            LEFT JOIN (
                SELECT 
                    si2.sale_id,
                    si2.product_id,
                    GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC) as serial_numbers
                FROM sale_items si2
                LEFT JOIN product_serial_numbers psn ON si2.product_id = psn.product_id 
                    AND psn.status = 'sold' 
                    AND psn.source_id = si2.sale_id
                    AND psn.source_type IN ('credit_sale', 'manual_adjustment')
                GROUP BY si2.sale_id, si2.product_id
            ) psn ON si.sale_id = psn.sale_id AND si.product_id = psn.product_id
            WHERE ct.customer_id = ? AND ct.id IN ($inClause)
            AND si.id NOT IN (
                SELECT DISTINCT ri.sale_item_id 
                FROM refund_items ri 
                JOIN refunds r ON ri.refund_id = r.id 
                WHERE r.sale_id = s.id
            )
            GROUP BY ct.id
            HAVING ct.amount - paid_amount > 0
        ";
        array_unshift($params, $customerId);
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $credits = $stmt->fetchAll();
        
        if (empty($credits)) {
            throw new Exception("No valid unpaid invoices found");
        }
        
        // Calculate total outstanding balance
        $totalOutstanding = 0;
        foreach ($credits as $credit) {
            $balance = $credit['amount'] - $credit['paid_amount'];
            $totalOutstanding += $balance;
        }
        
        // Store the previous balance before processing payment
        $previous_balance = $totalOutstanding;
        
        // Calculate total original credit amount for this customer (before any waivers)
        $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(ct.amount), 0) as total_original_credit
            FROM credit_transactions ct
            WHERE ct.customer_id = ?
        ");
        $stmt->execute([$customerId]);
        $totalOriginalCredit = $stmt->fetchColumn();
        
        // Validate payment amount
        if ($amount <= 0) {
            throw new Exception("Payment amount must be greater than 0");
        }
        
        if ($amount > $totalOutstanding) {
            throw new Exception("Payment amount cannot exceed total outstanding Debt balance of " . number_format($totalOutstanding, 2));
        }
        
        // Process payment
        $remainingAmount = $amount;
        $paymentDetails = [];
        $paymentDebug = [
            'old_balance' => $totalOutstanding,
            'paid_amount' => $amount,
            'new_balance' => $totalOutstanding - $amount,
            'payment_details' => []
        ];
        
        foreach ($credits as $credit) {
            if ($remainingAmount <= 0) break;
            
            $balance = $credit['amount'] - $credit['paid_amount'];
            if ($balance <= 0) continue;
            
            $paymentAmount = min($remainingAmount, $balance);
            
            // Insert payment record
            $stmt = $pdo->prepare("
                INSERT INTO credit_payments 
                (credit_transaction_id, sale_id, customer_id, amount, payment_method, reference, created_by) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $credit['id'],
                $credit['sale_id'],
                $customerId,
                $paymentAmount,
                $paymentMethod,
                $reference,
                $userId
            ]);
            
            // Get waived amount for this transaction
            $waiveStmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ? AND is_waived = 1");
            $waiveStmt->execute([$credit['id']]);
            $waived = $waiveStmt->fetchColumn();
            
            // Get detailed items for this sale
            $stmtItems = $pdo->prepare("
                SELECT si.*, p.name as product_name, 
                       GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC) as serial_numbers
                FROM sale_items si 
                JOIN products p ON si.product_id = p.id 
                LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id 
                    AND psn.status = 'sold' 
                    AND psn.source_id = si.sale_id
                    AND psn.source_type IN ('credit_sale', 'manual_adjustment')
                WHERE si.sale_id = ?
                AND si.id NOT IN (
                    SELECT DISTINCT ri.sale_item_id 
                    FROM refund_items ri 
                    JOIN refunds r ON ri.refund_id = r.id 
                    WHERE r.sale_id = si.sale_id
                )
                GROUP BY si.id, si.product_id, si.quantity, si.price, si.total, si.discount, si.unit_price, si.total_price, p.name
                ORDER BY p.name
            ");
            $stmtItems->execute([$credit['sale_id']]);
            $all_items = $stmtItems->fetchAll();

            // Build payment method details
            $paymentMethodDetails = '';
            if (!empty($credit['payment_methods'])) {
                $methods = explode(', ', $credit['payment_methods']);
                $amounts = explode(', ', $credit['payment_amounts']);
                $details = [];
                for ($i = 0; $i < count($methods); $i++) {
                    if (isset($amounts[$i]) && $amounts[$i] > 0) {
                        $details[] = ucfirst($methods[$i]) . ': ' . number_format($amounts[$i], 2);
                    }
                }
                $paymentMethodDetails = implode(', ', $details);
            } elseif (!empty($credit['sale_payment_method']) && $credit['sale_paid_amount'] > 0) {
                $paymentMethodDetails = ucfirst($credit['sale_payment_method']) . ': ' . number_format($credit['sale_paid_amount'], 2);
            } else {
                $paymentMethodDetails = 'Credit Sale';
            }
            
            $paymentDetails[] = [
                'invoice_number' => $credit['invoice_number'],
                'paid_amount' => $paymentAmount,
                'items' => $credit['items'], // Keep the concatenated string for backward compatibility
                'all_items' => $all_items, // Add detailed items array
                'total_quantity' => $credit['total_quantity'] ?? 0,
                'payment_method_details' => $paymentMethodDetails,
                'previously_waived' => $waived,
                'invoice_total' => $credit['amount'],
                'previously_paid' => $credit['paid_amount'],
                'balance_before' => $balance,
                'balance_after' => $balance - $paymentAmount
            ];
            
            $paymentDebug['payment_details'][] = [
                'invoice' => $credit['invoice_number'],
                'amount' => $paymentAmount,
                'balance_before' => $balance,
                'balance_after' => $balance - $paymentAmount
            ];
            
            $remainingAmount -= $paymentAmount;
        }
        
        // Calculate remaining balance after payment
        $remaining_balance = $previous_balance - $amount;
        
        // Log the payment
        $stmt = $pdo->prepare("
            INSERT INTO system_logs (user_id, action, details) 
            VALUES (?, 'credit_payment', ?)
        ");
        $logDetails = json_encode([
            'customer_id' => $customerId,
            'amount' => $amount,
            'payment_method' => $paymentMethod,
            'reference' => $reference,
            'invoices' => $paymentDetails
        ]);
        $stmt->execute([$userId, $logDetails]);
        
        $pdo->commit();
        
        $successMessage = "Payment of " . number_format($amount, 2) . " processed successfully";
        $_SESSION['success'] = $successMessage;
        $_SESSION['pay_debug'] = $paymentDebug;
        
        error_log("Payment successful - $successMessage");
        
        // Calculate total waived amount for this customer
        $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(cp.amount), 0) as total_waived
            FROM credit_payments cp
            WHERE cp.customer_id = ? AND cp.is_waived = 1
        ");
        $stmt->execute([$customerId]);
        $totalWaived = $stmt->fetchColumn();
        
        // Calculate total paid amount for this customer (excluding waivers)
        $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(cp.amount), 0) as total_paid
            FROM credit_payments cp
            WHERE cp.customer_id = ? AND (cp.is_waived = 0 OR cp.is_waived IS NULL)
        ");
        $stmt->execute([$customerId]);
        $totalPaid = $stmt->fetchColumn();
        
        // Get ALL credit invoices for this customer for complete receipt
        $stmtAllInvoices = $pdo->prepare("
            SELECT 
                ct.id,
                ct.sale_id,
                ct.amount,
                s.invoice_number,
                s.created_at as sale_date,
                COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
                ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance,
                CASE 
                    WHEN COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                         COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) >= ct.amount THEN 'Paid'
                    WHEN COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                         COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) > 0 THEN 'Partially Paid'
                    ELSE 'Pending'
                END as status
            FROM credit_transactions ct
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.customer_id = ?
            GROUP BY ct.id, ct.sale_id, ct.amount, s.invoice_number, s.created_at
            ORDER BY s.created_at DESC
        ");
        $stmtAllInvoices->execute([$customerId]);
        $allInvoices = $stmtAllInvoices->fetchAll();
        
        // Get items and payment details for each invoice
        $allInvoicesWithItems = [];
        foreach ($allInvoices as $invoice) {
            // Check if this invoice is being paid in current transaction
            $isBeingPaid = in_array($invoice['id'], $selectedInvoiceIds);
            $currentPaymentAmount = 0;
            
            if ($isBeingPaid) {
                // Find the payment amount for this invoice in current transaction
                foreach ($paymentDetails as $detail) {
                    if ($detail['invoice_number'] === $invoice['invoice_number']) {
                        $currentPaymentAmount = $detail['paid_amount'];
                        break;
                    }
                }
            }
            
            // Get all items for this invoice
            $stmtInvoiceItems = $pdo->prepare("
                SELECT si.*, p.name as product_name, 
                       GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC) as serial_numbers
                FROM sale_items si 
                JOIN products p ON si.product_id = p.id 
                LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id 
                    AND psn.status = 'sold' 
                    AND psn.source_id = si.sale_id
                    AND psn.source_type = 'credit_sale'
                WHERE si.sale_id = ?
                GROUP BY si.id, si.product_id, si.quantity, si.price, si.total, si.discount, si.unit_price, si.total_price, p.name
                ORDER BY p.name
            ");
            $stmtInvoiceItems->execute([$invoice['sale_id']]);
            $invoice_items = $stmtInvoiceItems->fetchAll();
            
            // Get payment method details for this invoice
            $stmtPaymentMethods = $pdo->prepare("SELECT method, amount FROM sale_payments WHERE sale_id = ? AND LOWER(method) != 'credit'");
            $stmtPaymentMethods->execute([$invoice['sale_id']]);
            $pos_payments = $stmtPaymentMethods->fetchAll();
            
            $pos_paid_amount = 0;
            $payment_methods = [];
            foreach ($pos_payments as $payment) {
                $pos_paid_amount += $payment['amount'];
                $payment_methods[] = ucfirst(strtolower($payment['method']));
            }
            
            $paymentMethodDetails = empty($payment_methods) ? 'Credit Sale' : implode(', ', $payment_methods);
            
            $allInvoicesWithItems[] = [
                'invoice_number' => $invoice['invoice_number'],
                'sale_date' => $invoice['sale_date'],
                'credit_amount' => $invoice['amount'],
                'previously_paid' => $invoice['paid'],
                'previously_waived' => $invoice['waived'],
                'current_payment' => $currentPaymentAmount,
                'balance_before' => $invoice['balance'] + $currentPaymentAmount,
                'balance_after' => $invoice['balance'],
                'status' => $invoice['status'],
                'items' => $invoice_items,
                'payment_method_details' => $paymentMethodDetails,
                'is_being_paid' => $isBeingPaid,
                'total_quantity' => array_sum(array_column($invoice_items, 'quantity'))
            ];
        }
        
        // Send payment confirmation notifications based on user preferences and settings
        try {
            // Get notification preferences from form
            $sendSMS = isset($_POST['send_sms']) && $_POST['send_sms'] === 'on' && ($settings['enable_sms'] ?? '0') === '1';
            $sendEmail = isset($_POST['send_email']) && $_POST['send_email'] === 'on' && ($settings['enable_email'] ?? '0') === '1';
            $sendWhatsApp = isset($_POST['send_whatsapp']) && $_POST['send_whatsapp'] === 'on' && ($settings['enable_whatsapp'] ?? '0') === '1';
            
            // Only proceed if at least one notification type is enabled
            if ($sendSMS || $sendEmail || $sendWhatsApp) {
                // Get customer contact information
                $stmt = $pdo->prepare("SELECT name, email, phone FROM customers WHERE id = ?");
                $stmt->execute([$customerId]);
                $customer = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($customer) {
                    // Include notification functions and templates
                    require_once '../includes/email_templates.php';
                    
                    // Initialize variables for tracking
                    $whatsappLinkGenerated = null;
                    
                    $notificationsSent = [];
                    $notificationErrors = [];
                    
                    // Send Email notification if enabled and email available
                    if ($sendEmail && !empty($customer['email'])) {
                        try {
                            // Email subject
                            $emailSubject = "Payment Confirmation - {$storeName}";
                            
                            // Log email attempt
                            error_log("Attempting to send payment confirmation email to: " . $customer['email']);
                            
                            // Create beautiful HTML email for payment confirmation
                            if (!function_exists('createDetailedPaymentConfirmationEmail')) {
                                throw new Exception('Email template function not available');
                            }
                            
                            // Format payment method for display
                            $displayPaymentMethod = $paymentMethod;
                            if ($paymentMethod === 'mpesa_received') {
                                $displayPaymentMethod = 'Sent Via M-PESA(Received)';
                            } elseif ($paymentMethod === 'mpesa') {
                                $displayPaymentMethod = 'M-Pesa';
                            } elseif ($paymentMethod === 'cash') {
                                $displayPaymentMethod = 'Cash';
                            }
                            
                            $emailMessage = createDetailedPaymentConfirmationEmail(
                                $customer['name'],
                                $amount,
                                $displayPaymentMethod,
                                $previous_balance,
                                $remaining_balance,
                                $totalPaid,
                                $paymentDetails,
                                $currency,
                                $storeName,
                                $storeAddress,
                                $storePhone,
                                $storeEmail,
                                $storeWebsite,
                                $showPaidInvoicesInMessages,
                                $reference
                            );
                            
                            // Verify email content was generated
                            if (empty($emailMessage)) {
                                throw new Exception('Email template generation failed');
                            }
                            
                            // Verify email function exists and is working
                            if (!function_exists('sendEmail')) {
                                throw new Exception('Email function not available');
                            }
                            
                            $emailResult = sendEmail($customer['email'], $emailSubject, $emailMessage);
                            error_log("Email sent successfully to: " . $customer['email']);
                            $notificationsSent[] = 'Email';
                        } catch (Exception $e) {
                            error_log("Email sending error: " . $e->getMessage());
                            $notificationErrors[] = 'Email: ' . $e->getMessage();
                        }
                    }
                    
                    // Send SMS/WhatsApp notifications if enabled and phone available
                    if (($sendSMS || $sendWhatsApp) && !empty($customer['phone'])) {
                        // Create payment confirmation message for SMS/WhatsApp
                        $paymentMessage = "Payment Confirmation - {$storeName}\n\n";
                        $paymentMessage .= "Dear {$customer['name']},\n\n";
                        $paymentMessage .= "We have received your payment of {$currency} " . number_format($amount, 2) . "\n";
                        // Format payment method for display
                        $displayPaymentMethod = $paymentMethod;
                        if ($paymentMethod === 'mpesa_received') {
                            $displayPaymentMethod = 'Sent Via M-PESA(Received)';
                        } elseif ($paymentMethod === 'mpesa') {
                            $displayPaymentMethod = 'M-Pesa';
                        } elseif ($paymentMethod === 'cash') {
                            $displayPaymentMethod = 'Cash';
                        }
                        
                        $paymentMessage .= "Payment Method: {$displayPaymentMethod}\n";
                        if (!empty($reference)) {
                            $paymentMessage .= "Reference: {$reference}\n";
                        }
                        $paymentMessage .= "Previous Balance: {$currency} " . number_format($previous_balance, 2) . "\n";
                        $paymentMessage .= "Amount Paid: {$currency} " . number_format($amount, 2) . "\n";
                        $paymentMessage .= "New Balance: {$currency} " . number_format($remaining_balance, 2) . "\n\n";
                        
                        // Show paid invoices only if configured to do so
                        if ($showPaidInvoicesInMessages && !empty($paymentDetails)) {
                            $paymentMessage .= "Paid Invoices:\n";
                            foreach ($paymentDetails as $detail) {
                                $paymentMessage .= "- INV-" . preg_replace('/[^0-9]/', '', $detail['invoice_number']) . 
                                                 ": {$currency} " . number_format($detail['paid_amount'], 2) . "\n";
                            }
                            $paymentMessage .= "\n";
                        }
                        
                        $paymentMessage .= "Thank you for your payment!\n\n";
                        $paymentMessage .= "Contact us: {$storePhone} | {$storeEmail}";
                        
                        // Send SMS notification if enabled
                        if ($sendSMS) {
                            try {
                                // Log SMS attempt
                                error_log("Attempting to send payment confirmation SMS to: " . $customer['phone']);
                                
                                // Verify SMS function exists and is working
                                if (!function_exists('sendSMS')) {
                                    throw new Exception('SMS function not available');
                                }
                                
                                $smsResult = sendSMS($customer['phone'], $paymentMessage);
                                error_log("SMS sent successfully to: " . $customer['phone']);
                                $notificationsSent[] = 'SMS';
                            } catch (Exception $e) {
                                error_log("SMS sending error: " . $e->getMessage());
                                $notificationErrors[] = 'SMS: ' . $e->getMessage();
                            }
                        }
                        
                        // Send WhatsApp notification if enabled (using link method)
                        if ($sendWhatsApp) {
                            try {
                                // Log WhatsApp attempt
                                error_log("Attempting to generate WhatsApp link for: " . $customer['phone']);
                                
                                // Verify WhatsApp function exists
                                if (!function_exists('generateWhatsAppLink')) {
                                    throw new Exception('WhatsApp function not available');
                                }
                                
                                // Format phone number for WhatsApp
                                $whatsappPhone = preg_replace('/[^0-9]/', '', $customer['phone']);
                                if (!empty($whatsappPhone)) {
                                    $whatsappLink = generateWhatsAppLink($customer['phone'], $paymentMessage);
                                    
                                    // Verify link was generated
                                    if (empty($whatsappLink)) {
                                        throw new Exception('WhatsApp link generation failed');
                                    }
                                    
                                    // Log the WhatsApp link generation
                                    error_log("WhatsApp link generated successfully: " . $whatsappLink);
                                    
                                    // Store the link for potential use (could be displayed to user or auto-opened)
                                    $whatsappLinkGenerated = $whatsappLink;
                                    $notificationsSent[] = 'WhatsApp Link';
                                } else {
                                    $notificationErrors[] = 'WhatsApp: Invalid phone number';
                                }
                            } catch (Exception $e) {
                                $notificationErrors[] = 'WhatsApp: ' . $e->getMessage();
                            }
                        }
                    }
                    
                    // Log notification results
                    if (!empty($notificationsSent)) {
                        error_log("Payment notifications sent successfully: " . implode(', ', $notificationsSent));
                    }
                    if (!empty($notificationErrors)) {
                        error_log("Payment notification errors: " . implode('; ', $notificationErrors));
                    }
                    
                    // Debug: Log notification configuration for troubleshooting
                    error_log("Notification Debug - SMS enabled: " . ($settings['enable_sms'] ?? '0') . 
                             ", Email enabled: " . ($settings['enable_email'] ?? '0') . 
                             ", WhatsApp enabled: " . ($settings['enable_whatsapp'] ?? '0'));
                    error_log("Customer contact info - Phone: " . ($customer['phone'] ?? 'N/A') . 
                             ", Email: " . ($customer['email'] ?? 'N/A'));
                }
            }
        } catch (Exception $e) {
            // Don't fail the payment if notifications fail, just log the error
            error_log("Payment notification error: " . $e->getMessage());
        }
        
        echo json_encode([
            'success' => true,
            'message' => $successMessage,
            'customer' => $customerName,
            'amount' => $amount,
            'payment_method' => $paymentMethod,
            'payment_details' => $paymentDetails,
            'all_invoices' => $allInvoicesWithItems,
            'previous_balance' => $previous_balance,
            'total_original_credit' => $totalOriginalCredit,
            'total_paid' => $totalPaid,
            'remaining_balance' => $remaining_balance,
            'waived_amount' => $totalWaived,
            'notifications_sent' => $notificationsSent ?? [],
            'notification_errors' => $notificationErrors ?? [],
            'whatsapp_link' => $whatsappLinkGenerated ?? null,
            'debug' => $paymentDebug
        ]);
        exit;
        
        } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        
        $errorMessage = "Error processing payment: " . $e->getMessage();
        error_log("Payment error: $errorMessage");
        $_SESSION['error'] = $errorMessage;
        
                echo json_encode([
                    'success' => false,
            'message' => $errorMessage
                ]);
                exit;
    }
}

// Handle reminder action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reminder_id'])) {
    header('Content-Type: application/json');
    
    try {
        $reminderId = $_POST['reminder_id'];
        $reminderType = $_POST['reminder_type'];
        $status = ['success' => true, 'message' => ''];
        
        // Get credit transaction and customer details with waiver information
        $stmt = $pdo->prepare("
            SELECT ct.*, c.name as customer_name, c.email, c.phone, 
                   s.invoice_number, s.created_at as sale_date,
                   COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid_amount,
                   COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived_amount,
                   ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
                   COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as remaining_balance
            FROM credit_transactions ct
            JOIN customers c ON ct.customer_id = c.id
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.id = ?
            GROUP BY ct.id
        ");
        $stmt->execute([$reminderId]);
        $credit = $stmt->fetch();
        
        if (!$credit) {
            throw new Exception("Credit transaction not found");
        }

        if ($credit['remaining_balance'] <= 0) {
            throw new Exception("No remaining balance for this transaction");
        }

        $remainingBalance = number_format($credit['remaining_balance'], 2);
        
        // Get sale items for this transaction
        $stmt = $pdo->prepare("
            SELECT p.name, si.quantity, si.price
            FROM sale_items si
            JOIN products p ON si.product_id = p.id
            WHERE si.sale_id = ?
            ORDER BY p.name
        ");
        $stmt->execute([$credit['sale_id']]);
        $saleItems = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Include email templates
        require_once '../includes/email_templates.php';
        
        // Create beautiful HTML email with waiver information
        $message = createCreditReminderEmail(
            $credit['customer_name'],
            $credit['invoice_number'],
            $credit['sale_date'],
            $credit['remaining_balance'],
            $currency,
            $storeName,
            $storeAddress,
            $storePhone,
            $storeEmail,
            $storeWebsite,
            $saleItems,
            $credit['customer_id'],
            $credit['amount'], // original credit amount
            $credit['paid_amount'], // paid amount
            $credit['waived_amount'] // waived amount
        );
        
        $subject = "Payment Reminder - {$storeName}";
        
        // Log the reminder attempt
        $logDetails = json_encode([
            'customer_id' => $credit['customer_id'],
            'customer_name' => $credit['customer_name'],
            'invoice_number' => $credit['invoice_number'],
            'remaining_balance' => $remainingBalance,
            'reminder_type' => $reminderType
        ]);
        
        $stmt = $pdo->prepare("
            INSERT INTO system_logs (user_id, action, details) 
            VALUES (?, 'credit_reminder', ?)
        ");
        $stmt->execute([$_SESSION['user_id'], $logDetails]);
        
        $sentMessages = [];
        
        if ($reminderType === 'sms' || $reminderType === 'both') {
            if (empty($credit['phone'])) {
                $status['message'] .= "SMS not sent: Customer has no phone number configured. ";
            } else {
                $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'sms_settings'");
                $stmt->execute();
                $smsSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
                
                if (($smsSettings['enable_sms'] ?? '0') !== '1') {
                    $status['message'] .= "SMS not sent: SMS notifications are not enabled. ";
                } elseif (empty($smsSettings['sms_api_key']) || empty($smsSettings['sms_api_secret']) || empty($smsSettings['sms_sender_id'])) {
                    $status['message'] .= "SMS not sent: SMS settings are not properly configured. ";
                } else {
                    try {
                        $result = sendSMS($credit['phone'], $message);
                        $sentMessages[] = 'SMS';
                    } catch (Exception $e) {
                        $status['message'] .= "SMS not sent: " . $e->getMessage() . ". ";
                    }
                }
            }
        }
        
        if ($reminderType === 'email' || $reminderType === 'both') {
            if (empty($credit['email'])) {
                $status['message'] .= "Email not sent: Customer has no email address configured. ";
            } else {
                $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'email_settings'");
                $stmt->execute();
                $emailSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
                
                if (($emailSettings['enable_email'] ?? '0') !== '1') {
                    $status['message'] .= "Email not sent: Email notifications are not enabled. ";
                } elseif (empty($emailSettings['smtp_host']) || empty($emailSettings['smtp_port']) || 
                         empty($emailSettings['smtp_username']) || empty($emailSettings['smtp_password'])) {
                    $status['message'] .= "Email not sent: Email settings are not properly configured. ";
                } else {
                    try {
                        $result = sendEmail($credit['email'], $subject, $message);
                        $sentMessages[] = 'Email';
                        error_log("Email reminder sent successfully to: " . $credit['email']);
                    } catch (Exception $e) {
                        $errorMessage = $e->getMessage();
                        error_log("Email reminder failed for " . $credit['email'] . ": " . $errorMessage);
                        $status['message'] .= "Email not sent: " . $errorMessage . ". ";
                    }
                }
            }
        }
        
        if (!empty($sentMessages)) {
            $status['message'] = "Successfully sent " . implode(' and ', $sentMessages) . " reminder(s). " . $status['message'];
        } else {
            $status['success'] = false;
            if (empty($status['message'])) {
                $status['message'] = "Failed to send any reminders.";
            }
        }
        
        echo json_encode($status);
        exit;
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Reminder error: ' . $e->getMessage()
        ]);
        exit;
    }
}

// Handle waive action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['waive_customer_id'])) {
    header('Content-Type: application/json');
    
    try {
        $customerId = $_POST['waive_customer_id'];
        $amount = floatval($_POST['waive_amount']);
        $reason = $_POST['waive_reason'];
        $userId = $_SESSION['user_id'];
        $selectedInvoiceIds = isset($_POST['waive_invoice_ids']) ? array_map('intval', (array)$_POST['waive_invoice_ids']) : [];

        if (empty($selectedInvoiceIds)) {
            echo json_encode([
                'success' => false,
                'message' => "Please select at least one unpaid invoice to waive."
            ]);
            exit;
        }

        // Get customer details
        $stmt = $pdo->prepare("SELECT name FROM customers WHERE id = ?");
        $stmt->execute([$customerId]);
        $customerName = $stmt->fetchColumn();
        
        if (!$customerName) {
            echo json_encode([
                'success' => false,
                'message' => "Customer not found"
            ]);
            exit;
        }

        // Only fetch and process selected invoices with detailed unpaid credit calculation
        $inClause = implode(',', array_fill(0, count($selectedInvoiceIds), '?'));
        $params = $selectedInvoiceIds;
        $sql = "
            SELECT 
                ct.id, 
                ct.sale_id, 
                ct.amount, 
                   s.invoice_number,
                s.created_at as sale_date,
                COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
                ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance,
                GROUP_CONCAT(
                    DISTINCT CONCAT(p.name, 
                        CASE 
                            WHEN psn.serial_numbers IS NOT NULL AND psn.serial_numbers != '' 
                            THEN CONCAT(' [SN: ', psn.serial_numbers, ']') 
                            ELSE '' 
                        END
                    ) ORDER BY p.name SEPARATOR ', '
                ) as items,
                COALESCE(SUM(si.quantity), 0) as total_quantity,
                CASE 
                    WHEN COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                         COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) >= ct.amount THEN 'Paid'
                    WHEN COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                         COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) > 0 THEN 'Partially Paid'
                    ELSE 'Pending'
                END as status
            FROM credit_transactions ct
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            LEFT JOIN sale_items si ON s.id = si.sale_id
            LEFT JOIN products p ON si.product_id = p.id
            LEFT JOIN (
                SELECT 
                    si2.sale_id,
                    si2.product_id,
                    GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC) as serial_numbers
                FROM sale_items si2
                LEFT JOIN product_serial_numbers psn ON si2.product_id = psn.product_id 
                    AND psn.status = 'sold' 
                    AND psn.source_id = si2.sale_id
                    AND psn.source_type = 'credit_sale'
                GROUP BY si2.sale_id, si2.product_id
            ) psn ON si.sale_id = psn.sale_id AND si.product_id = psn.product_id
            WHERE ct.customer_id = ? AND ct.id IN ($inClause)
            GROUP BY ct.id, ct.sale_id, ct.amount, s.invoice_number, s.created_at
            HAVING balance > 0
        ";
        array_unshift($params, $customerId);
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $credits = $stmt->fetchAll();

        // Calculate old balance BEFORE waiving using improved unpaid credit calculation
        $old_balance = 0;
        $totalOriginalCredit = 0;
        foreach ($credits as $credit) {
            $itemBalance = $credit['balance']; // Use the calculated balance from the query
            if ($itemBalance > 0) {
                $old_balance += $itemBalance;
            }
            $totalOriginalCredit += $credit['amount']; // Add original credit amount
        }

        // Validate
        if ($amount <= 0) {
            echo json_encode([
                'success' => false,
                'message' => "Waive amount must be greater than 0"
            ]);
            exit;
        }

        if ($amount > $old_balance) {
            echo json_encode([
                'success' => false,
                'message' => "Waive amount cannot exceed outstanding balance of selected invoices."
            ]);
            exit;
        }

        $new_balance = $old_balance - $amount;
        $waive_details = [];
        $waive_debug_details = [];
        $remainingAmount = $amount;
        $seen = [];

        $pdo->beginTransaction();

        foreach ($credits as $credit) {
            if (in_array($credit['id'], $seen)) continue;
            $seen[] = $credit['id'];
            $itemBalance = $credit['balance']; // Use the calculated balance from the query
            if ($itemBalance > 0 && $remainingAmount > 0) {
                $waiveNow = min($remainingAmount, $itemBalance);
                $stmtWaive = $pdo->prepare("
                    INSERT INTO credit_payments 
                    (credit_transaction_id, sale_id, customer_id, amount, payment_method, reference, created_by, is_waived, waive_reason) 
                    VALUES (?, ?, ?, ?, 'cash', ?, ?, 1, ?)
                ");
                $stmtWaive->execute([
                    $credit['id'],
                    $credit['sale_id'],
                    $customerId,
                    $waiveNow,
                    'waiveD-' . date('YmdHis'),
                    $userId,
                    $reason
                ]);
                
                // Get POS payment details for this sale (non-credit payments)
                $stmtPOS = $pdo->prepare("SELECT method, amount FROM sale_payments WHERE sale_id = ? AND LOWER(method) != 'credit'");
                $stmtPOS->execute([$credit['sale_id']]);
                $pos_payments = $stmtPOS->fetchAll();
                
                // Calculate total POS paid amount and payment methods
                $pos_paid_amount = 0;
                $payment_methods = [];
                foreach ($pos_payments as $payment) {
                    $pos_paid_amount += $payment['amount'];
                    $payment_methods[] = ucfirst(strtolower($payment['method']));
                }
                
                // Get ALL items from this sale (not just waived items)
                $stmtItems = $pdo->prepare("
                    SELECT si.*, p.name as product_name, 
                           GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC) as serial_numbers
                    FROM sale_items si 
                    JOIN products p ON si.product_id = p.id 
                    LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id 
                        AND psn.status = 'sold' 
                        AND psn.source_id = si.sale_id
                        AND psn.source_type = 'credit_sale'
                    WHERE si.sale_id = ?
                    GROUP BY si.id, si.product_id, si.quantity, si.price, si.total, si.discount, si.unit_price, si.total_price, p.name
                    ORDER BY p.name
                ");
                $stmtItems->execute([$credit['sale_id']]);
                $all_items = $stmtItems->fetchAll();
                
                // Format items list with quantities and prices
                $formatted_items = [];
                foreach ($all_items as $item) {
                    $item_text = $item['product_name'] . ' (Qty: ' . $item['quantity'] . ' @ ' . number_format($item['price'], 2) . ')';
                    if (!empty($item['serial_numbers'])) {
                        $serial_numbers = explode(',', $item['serial_numbers']);
                        $item_text .= ' [SN: ' . implode(', ', $serial_numbers) . ']';
                    }
                    $formatted_items[] = $item_text;
                }
                $all_items_text = implode(', ', $formatted_items);
                
                // Build payment method details like in Credit Items table
                $paymentMethodDetails = '';
                if (!empty($payment_methods)) {
                    $details = [];
                    foreach ($payment_methods as $method) {
                        if ($pos_paid_amount > 0) {
                            $details[] = $method . ': ' . number_format($pos_paid_amount, 2);
                        }
                    }
                    $paymentMethodDetails = implode(', ', $details);
                } else {
                    $paymentMethodDetails = 'Credit Sale';
                }
                
                // Get total quantity for this sale
                $stmtQty = $pdo->prepare("SELECT COALESCE(SUM(quantity), 0) FROM sale_items WHERE sale_id = ?");
                $stmtQty->execute([$credit['sale_id']]);
                $totalQuantity = $stmtQty->fetchColumn();
                
                // Enhanced waive details with more information like sales receipts
                $waive_details[] = [
                    'invoice_number' => $credit['invoice_number'],
                    'sale_date' => $credit['sale_date'],
                    'invoice_total' => $credit['amount'],
                    'previously_paid' => $credit['paid'],
                    'previously_waived' => $credit['waived'],
                    'balance_before' => $itemBalance,
                    'waived_amount' => $waiveNow,
                    'balance_after' => $itemBalance - $waiveNow,
                    'items' => $all_items_text, // Use the detailed formatted items list
                    'total_quantity' => $totalQuantity,
                    'payment_method_details' => $paymentMethodDetails,
                    'all_items' => $all_items, // Include the full items array for detailed display
                    'status' => $credit['status'],
                    'pos_paid_amount' => $pos_paid_amount,
                    'payment_methods' => $payment_methods
                ];
                $waive_debug_details[] = 'Invoice ' . $credit['invoice_number'] . ': Waived ' . $waiveNow;
                $remainingAmount -= $waiveNow;
            }
        }

        $pdo->commit();

        // After waiving, recalculate the sum of balances for all invoices for this customer using improved calculation
        $stmt = $pdo->prepare("
            SELECT 
                ct.amount,
                COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
                ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance
            FROM credit_transactions ct
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.customer_id = ?
            GROUP BY ct.id, ct.amount
        ");
        $stmt->execute([$customerId]);
        $all_credits = $stmt->fetchAll();
        $remaining_balance = 0;
        foreach ($all_credits as $c) {
            if ($c['balance'] > 0.001) $remaining_balance += $c['balance'];
        }

        // Get ALL unpaid credit invoices for this customer (for complete receipt)
        $stmtAllUnpaid = $pdo->prepare("
            SELECT 
                ct.id,
                ct.sale_id,
                ct.amount,
                s.invoice_number,
                s.created_at as sale_date,
                COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
                ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance,
                CASE 
                    WHEN COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                         COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) >= ct.amount THEN 'Paid'
                    WHEN COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                         COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) > 0 THEN 'Partially Paid'
                    ELSE 'Pending'
                END as status
            FROM credit_transactions ct
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.customer_id = ?
            GROUP BY ct.id, ct.sale_id, ct.amount, s.invoice_number, s.created_at
            HAVING balance > 0
            ORDER BY s.created_at DESC
        ");
        $stmtAllUnpaid->execute([$customerId]);
        $all_unpaid_invoices = $stmtAllUnpaid->fetchAll();
        
        // Get items and payment details for each unpaid invoice
        $unpaid_invoices_with_items = [];
        foreach ($all_unpaid_invoices as $invoice) {
            // Get all items for this invoice
            $stmtInvoiceItems = $pdo->prepare("
                SELECT si.*, p.name as product_name, 
                       GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC) as serial_numbers
                FROM sale_items si 
                JOIN products p ON si.product_id = p.id 
                LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id 
                    AND psn.status = 'sold' 
                    AND psn.source_id = si.sale_id
                    AND psn.source_type = 'credit_sale'
                WHERE si.sale_id = ?
                GROUP BY si.id, si.product_id, si.quantity, si.price, si.total, si.discount, si.unit_price, si.total_price, p.name
                ORDER BY p.name
            ");
            $stmtInvoiceItems->execute([$invoice['sale_id']]);
            $invoice_items = $stmtInvoiceItems->fetchAll();
            
            // Get POS payments for this invoice
            $stmtInvoicePOS = $pdo->prepare("SELECT method, amount FROM sale_payments WHERE sale_id = ? AND LOWER(method) != 'credit'");
            $stmtInvoicePOS->execute([$invoice['sale_id']]);
            $invoice_pos_payments = $stmtInvoicePOS->fetchAll();
            
            $invoice_pos_paid = 0;
            $invoice_payment_methods = [];
            foreach ($invoice_pos_payments as $payment) {
                $invoice_pos_paid += $payment['amount'];
                $invoice_payment_methods[] = ucfirst(strtolower($payment['method']));
            }
            
            $unpaid_invoices_with_items[] = [
                'invoice_number' => $invoice['invoice_number'],
                'sale_date' => $invoice['sale_date'],
                'credit_amount' => $invoice['amount'],
                'paid' => $invoice['paid'],
                'waived' => $invoice['waived'],
                'balance' => $invoice['balance'],
                'status' => $invoice['status'],
                'items' => $invoice_items,
                'pos_paid_amount' => $invoice_pos_paid,
                'payment_methods' => $invoice_payment_methods
            ];
        }

        echo json_encode([
            'success' => true,
            'message' => 'Amount waived successfully',
            'old_balance' => $old_balance,
            'total_original_credit' => $totalOriginalCredit,
            'waived_amount' => $amount,
            'new_balance' => $new_balance,
            'waive_details' => $waive_details,
            'remaining_balance' => $remaining_balance,
            'all_unpaid_invoices' => $unpaid_invoices_with_items
        ]);
        exit;

    } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        
        echo json_encode([
            'success' => false,
            'message' => "Error waiving amount: " . $e->getMessage()
        ]);
        exit;
    }
}

// Handle edit invoice action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['edit_invoice_id'])) {
    header('Content-Type: application/json');
    
    try {
        $invoiceId = intval($_POST['edit_invoice_id']);
        $editReason = $_POST['edit_reason'] ?? '';
        $userId = $_SESSION['user_id'];
        
        if (empty($editReason)) {
            echo json_encode([
                'success' => false,
                'message' => "Please provide a reason for editing the invoice."
            ]);
            exit;
        }
        
        // Get invoice details
        $stmt = $pdo->prepare("
            SELECT s.*, c.name as customer_name, ct.id as credit_transaction_id
            FROM sales s 
            JOIN credit_transactions ct ON s.id = ct.sale_id
            JOIN customers c ON ct.customer_id = c.id
            WHERE s.id = ?
        ");
        $stmt->execute([$invoiceId]);
        $invoice = $stmt->fetch();
        
        if (!$invoice) {
            echo json_encode([
                'success' => false,
                'message' => "Invoice not found."
            ]);
            exit;
        }
        
        $pdo->beginTransaction();
        
        // Get the updated items from POST data
        $updatedItems = $_POST['items'] ?? [];
        
        if (empty($updatedItems)) {
            echo json_encode([
                'success' => false,
                'message' => "No items to update."
            ]);
            exit;
        }
        
        $newTotal = 0;
        $originalTotal = $invoice['total_amount'];
        
        // Update each sale item
        foreach ($updatedItems as $itemId => $itemData) {
            $newQty = intval($itemData['quantity']);
            $newPrice = floatval($itemData['price']);
            $newItemTotal = $newQty * $newPrice;
            $newTotal += $newItemTotal;
            
            // Update sale_items table
            $stmt = $pdo->prepare("
                UPDATE sale_items 
                SET quantity = ?, price = ?, total = ?, total_price = ?
                WHERE id = ? AND sale_id = ?
            ");
            $stmt->execute([$newQty, $newPrice, $newItemTotal, $newItemTotal, $itemId, $invoiceId]);
        }
        
        // Update sales table with new total
        $stmt = $pdo->prepare("
            UPDATE sales 
            SET total_amount = ?, updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$newTotal, $invoiceId]);
        
        // Update credit_transactions table with new amount
        $stmt = $pdo->prepare("
            UPDATE credit_transactions 
            SET amount = ?
            WHERE sale_id = ?
        ");
        $stmt->execute([$newTotal, $invoiceId]);
        
        // Log the edit action
        $stmt = $pdo->prepare("
            INSERT INTO invoice_edits 
            (invoice_id, user_id, original_total, new_total, reason, created_at) 
            VALUES (?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([$invoiceId, $userId, $originalTotal, $newTotal, $editReason]);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Invoice updated successfully',
            'invoice_number' => $invoice['invoice_number'],
            'original_total' => $originalTotal,
            'new_total' => $newTotal,
            'difference' => $newTotal - $originalTotal
        ]);
        exit;
        
    } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        
        echo json_encode([
            'success' => false,
            'message' => "Error updating invoice: " . $e->getMessage()
        ]);
        exit;
    }
}

// Handle fetch invoice for edit
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['fetch_invoice_for_edit'])) {
    header('Content-Type: application/json');
    
    try {
        $customerId = intval($_GET['customer_id']);
        
        // Get unpaid invoices for this customer
        $stmt = $pdo->prepare("
            SELECT 
                s.id,
                s.invoice_number,
                s.total_amount,
                s.created_at,
                ct.amount as credit_amount,
                COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
                ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance
            FROM sales s
            JOIN credit_transactions ct ON s.id = ct.sale_id
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.customer_id = ?
            GROUP BY s.id, ct.id
            HAVING balance > 0
            ORDER BY s.created_at DESC
        ");
        $stmt->execute([$customerId]);
        $invoices = $stmt->fetchAll();
        
        echo json_encode([
            'success' => true,
            'invoices' => $invoices
        ]);
        exit;
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => "Error fetching invoices: " . $e->getMessage()
        ]);
        exit;
    }
}

// Handle fetch invoice items for edit
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['fetch_invoice_items'])) {
    header('Content-Type: application/json');
    
    try {
        $invoiceId = intval($_GET['invoice_id']);
        
        // Get invoice items
        $stmt = $pdo->prepare("
            SELECT 
                si.id,
                si.product_id,
                si.quantity,
                si.price,
                si.total,
                p.name as product_name
            FROM sale_items si
            JOIN products p ON si.product_id = p.id
            WHERE si.sale_id = ?
            ORDER BY p.name
        ");
        $stmt->execute([$invoiceId]);
        $items = $stmt->fetchAll();
        
        // Get invoice total
        $stmt = $pdo->prepare("SELECT total_amount FROM sales WHERE id = ?");
        $stmt->execute([$invoiceId]);
        $totalAmount = $stmt->fetchColumn();
        
        echo json_encode([
            'success' => true,
            'items' => $items,
            'total_amount' => $totalAmount
        ]);
        exit;
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => "Error fetching invoice items: " . $e->getMessage()
        ]);
        exit;
    }
}

// Get filter parameters
$status = $_GET['status'] ?? '';
$customer_id = $_GET['customer_id'] ?? '';
$search = $_GET['search'] ?? '';

// Build query
$query = "SELECT ct.*, s.invoice_number, s.created_at as sale_date, 
          c.name as customer_name,
          GROUP_CONCAT(DISTINCT
              CONCAT(p.name, 
                  CASE 
                      WHEN psn.serial_numbers IS NOT NULL AND psn.serial_numbers != '' 
                      THEN CONCAT(' [SN: ', psn.serial_numbers, ']') 
                      ELSE '' 
                  END
              ) SEPARATOR ', '
          ) as items,
          SUM(si.quantity) as total_quantity,
          COALESCE(SUM(CASE WHEN cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) as amount_paid,
          s.payment_method as sale_payment_method,
          s.paid_amount as sale_paid_amount,
          (SELECT GROUP_CONCAT(DISTINCT sp2.method ORDER BY sp2.id SEPARATOR ', ') FROM sale_payments sp2 WHERE sp2.sale_id = s.id) as payment_methods,
          (SELECT GROUP_CONCAT(DISTINCT sp2.amount ORDER BY sp2.id SEPARATOR ', ') FROM sale_payments sp2 WHERE sp2.sale_id = s.id) as payment_amounts,
          CASE 
              WHEN EXISTS (SELECT 1 FROM credit_payments WHERE credit_transaction_id = ct.id AND payment_method != 'refund') THEN 'paid'
              ELSE 'pending'
          END as status
          FROM credit_transactions ct
          JOIN sales s ON ct.sale_id = s.id
          JOIN customers c ON ct.customer_id = c.id
          LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
          LEFT JOIN sale_items si ON s.id = si.sale_id
          LEFT JOIN products p ON si.product_id = p.id
          LEFT JOIN (
              SELECT 
                  si2.sale_id,
                  si2.product_id,
                  GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC) as serial_numbers
              FROM sale_items si2
              LEFT JOIN product_serial_numbers psn ON si2.product_id = psn.product_id 
                  AND psn.status = 'sold' 
                  AND psn.source_id = si2.sale_id
                  AND psn.source_type IN ('credit_sale', 'manual_adjustment')
              GROUP BY si2.sale_id, si2.product_id
          ) psn ON si.sale_id = psn.sale_id AND si.product_id = psn.product_id
          WHERE 1=1
          AND si.id NOT IN (
              SELECT DISTINCT ri.sale_item_id 
              FROM refund_items ri 
              JOIN refunds r ON ri.refund_id = r.id 
              WHERE r.sale_id = s.id
          )";

$params = [];

if ($status) {
    if ($status === 'paid') {
        $query .= " AND EXISTS (SELECT 1 FROM credit_payments WHERE credit_transaction_id = ct.id AND payment_method != 'refund')";
    } else if ($status === 'pending') {
        $query .= " AND NOT EXISTS (SELECT 1 FROM credit_payments WHERE credit_transaction_id = ct.id AND payment_method != 'refund')";
    }
}

if ($customer_id) {
    $query .= " AND ct.customer_id = ?";
    $params[] = $customer_id;
}

if ($search) {
    $query .= " AND (s.invoice_number LIKE ? OR c.name LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$query .= " GROUP BY ct.id ORDER BY s.created_at DESC";

// Get credit transactions
$stmt = $pdo->prepare($query);
$stmt->execute($params);
$credits = $stmt->fetchAll();

// Get customers for filter
$stmt = $pdo->prepare("SELECT id, name FROM customers ORDER BY name");
$stmt->execute();
$customers = $stmt->fetchAll();

// Calculate statistics
$total_credit = 0;
$total_paid = 0;
$total_pending = 0;
$total_customers = 0;
$customers_with_credit = [];

// Get all statistics in a single query using improved unpaid credit calculation (including POS payments)
$stmt = $pdo->prepare("
    WITH credit_summary AS (
        SELECT 
            ct.customer_id,
            ct.amount as original_amount,
            -- Credit payments from credit_payments table (only actual payments toward credit debt, excluding refunds)
            COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) as amount_paid,
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as amount_waived,
            -- Subtract refunds from the original amount
            COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as amount_refunded,
            ct.amount - COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as remaining_balance
        FROM credit_transactions ct
        LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
        GROUP BY ct.id, ct.customer_id, ct.amount
        HAVING remaining_balance > 0
    )
    SELECT 
        COUNT(DISTINCT customer_id) as total_customers,
        SUM(original_amount - amount_refunded) as total_original_credit,
        SUM(amount_paid) as total_paid,
        SUM(amount_waived) as total_waived,
        SUM(remaining_balance) as total_pending
    FROM credit_summary
");
$stmt->execute();
$totals = $stmt->fetch(PDO::FETCH_ASSOC);

$total_customers = $totals['total_customers'] ?? 0;
$total_credit = $totals['total_original_credit'] ?? 0;
$total_paid = $totals['total_paid'] ?? 0;
$total_pending = $totals['total_pending'] ?? 0;

// Group credits by customer
$credits_by_customer = [];
$cleared_credits_by_customer = [];
foreach ($credits as $credit) {
    $cid = $credit['customer_id'];
    // Calculate paid and waived for this transaction using improved unpaid credit calculation (credit payments only, excluding refunds)
    $stmtBalance = $pdo->prepare("
        SELECT 
            ct.amount,
            -- Credit payments from credit_payments table (only actual payments toward credit debt, excluding refunds)
            COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) as paid,
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
            -- Subtract refunds from the original amount
            COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as refunded,
            ct.amount - COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as balance
        FROM credit_transactions ct
        LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
        WHERE ct.id = ?
        GROUP BY ct.id, ct.amount
    ");
    $stmtBalance->execute([$credit['id']]);
    $balanceData = $stmtBalance->fetch();
    $paid = $balanceData['paid'] ?? 0;
    $waived = $balanceData['waived'] ?? 0;
    $balance = $balanceData['balance'] ?? 0;

    // Only include invoices with a positive balance
    if ($balance > 0.001) {
        if (!isset($credits_by_customer[$cid])) {
            $credits_by_customer[$cid] = [
                'customer_name' => $credit['customer_name'] ?? $credit['name'] ?? 'Unknown',
                'total_credit' => 0,
                'total_paid' => 0,
                'total_waived' => 0,
                'total_balance' => 0,
                'status' => 'Not Paid',
                'transactions' => []
            ];
        }
        $credits_by_customer[$cid]['total_credit'] += ($credit['amount'] - $balanceData['refunded']);
        $credits_by_customer[$cid]['total_paid'] += $paid;
        $credits_by_customer[$cid]['total_waived'] += $waived;
        $credits_by_customer[$cid]['total_balance'] += $balance;
        $credit['amount_paid'] = $paid;
        $credit['amount_waived'] = $waived;
        $credit['balance'] = $balance;
        $credits_by_customer[$cid]['transactions'][] = $credit;
    }
}
// Set status for each customer
foreach ($credits_by_customer as &$group) {
    if ($group['total_paid'] == 0 && $group['total_waived'] == 0) {
        $group['status'] = 'Not Paid';
    } elseif ($group['total_paid'] > 0 && $group['total_paid'] < $group['total_credit']) {
        $group['status'] = 'Partially Paid';
    } elseif ($group['total_waived'] > 0 && $group['total_paid'] == 0) {
        $group['status'] = 'Waived';
    } else {
        $group['status'] = 'Partially Paid';
    }
}
unset($group);
// Separate cleared credits (total_balance == 0)
foreach ($credits_by_customer as $cid => $group) {
    if ($group['total_balance'] == 0) {
        $cleared_credits_by_customer[$cid] = $group;
        unset($credits_by_customer[$cid]);
    }
}

// --- Fetch all credit transactions for grouping cleared credits ---
$all_credits_query = "SELECT ct.*, s.invoice_number, s.created_at as sale_date, 
    c.name as customer_name,
    GROUP_CONCAT(DISTINCT
        CONCAT(p.name, 
            CASE 
                WHEN psn.serial_numbers IS NOT NULL AND psn.serial_numbers != '' 
                THEN CONCAT(' [SN: ', psn.serial_numbers, ']') 
                ELSE '' 
            END
        ) SEPARATOR ', '
    ) as items,
    (SELECT COALESCE(SUM(si2.quantity), 0) FROM sale_items si2 WHERE si2.sale_id = s.id AND si2.id NOT IN (
        SELECT DISTINCT ri.sale_item_id 
        FROM refund_items ri 
        JOIN refunds r ON ri.refund_id = r.id 
        WHERE r.sale_id = s.id
    )) as total_quantity,
    s.payment_method as sale_payment_method,
    s.paid_amount as sale_paid_amount,
    GROUP_CONCAT(DISTINCT sp.method ORDER BY sp.id SEPARATOR ', ') as payment_methods,
    GROUP_CONCAT(DISTINCT sp.amount ORDER BY sp.id SEPARATOR ', ') as payment_amounts,
    -- Credit payments from credit_payments table (only actual payments toward credit debt, excluding refunds)
    COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) as amount_paid,
    -- Subtract refunds from the original amount
    COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as amount_refunded,
    CASE 
        WHEN (
            COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) +
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) +
            COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0)
        ) >= ct.amount THEN 'paid'
        ELSE 'pending'
    END as status
    FROM credit_transactions ct
    JOIN sales s ON ct.sale_id = s.id
    JOIN customers c ON ct.customer_id = c.id
    LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
    LEFT JOIN sale_items si ON s.id = si.sale_id
    LEFT JOIN products p ON si.product_id = p.id
    LEFT JOIN (
        SELECT 
            si2.sale_id,
            si2.product_id,
            GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC) as serial_numbers
        FROM sale_items si2
        LEFT JOIN product_serial_numbers psn ON si2.product_id = psn.product_id 
            AND psn.status = 'sold' 
            AND psn.source_id = si2.sale_id
            AND psn.source_type = 'credit_sale'
        GROUP BY si2.sale_id, si2.product_id
    ) psn ON si.sale_id = psn.sale_id AND si.product_id = psn.product_id
    LEFT JOIN sale_payments sp ON s.id = sp.sale_id
    WHERE si.id NOT IN (
        SELECT DISTINCT ri.sale_item_id 
        FROM refund_items ri 
        JOIN refunds r ON ri.refund_id = r.id 
        WHERE r.sale_id = s.id
    )
    GROUP BY ct.id ORDER BY s.created_at DESC";
$all_credits_stmt = $pdo->prepare($all_credits_query);
$all_credits_stmt->execute();
$all_credits = $all_credits_stmt->fetchAll();

// Group all credits by customer for cleared credits
$all_credits_by_customer = [];
$cleared_credits_by_customer = [];
foreach ($all_credits as $credit) {
    $cid = $credit['customer_id'];
    if (!isset($all_credits_by_customer[$cid])) {
        $all_credits_by_customer[$cid] = [
            'customer_name' => $credit['customer_name'] ?? $credit['name'] ?? 'Unknown',
            'total_credit' => 0,
            'total_paid' => 0,
            'total_waived' => 0,
            'total_balance' => 0,
            'transactions' => []
        ];
    }
    $paid = $credit['amount_paid'];
    // Calculate waived for this transaction
    $waiveStmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ? AND is_waived = 1");
    $waiveStmt->execute([$credit['id']]);
    $waived = $waiveStmt->fetchColumn();
    $all_credits_by_customer[$cid]['total_credit'] += ($credit['amount'] - $credit['amount_refunded']);
    $all_credits_by_customer[$cid]['total_paid'] += $paid;
    $all_credits_by_customer[$cid]['total_waived'] += $waived;
    $all_credits_by_customer[$cid]['total_balance'] += ($credit['amount'] - $credit['amount_refunded']) - ($paid + $waived);
    $credit['amount_waived'] = $waived;
    $all_credits_by_customer[$cid]['transactions'][] = $credit;
}
// Separate cleared credits (total_balance == 0)
foreach ($all_credits_by_customer as $cid => $group) {
    if ($group['total_credit'] > 0 && abs($group['total_balance']) < 0.01) {
        $cleared_credits_by_customer[$cid] = $group;
    }
}

if (isset($_GET['fetch_unpaid_invoices']) && $_GET['fetch_unpaid_invoices'] == '1' && isset($_GET['customer_id'])) {
    $customerId = intval($_GET['customer_id']);
            $stmt = $pdo->prepare("
            SELECT ct.id, s.invoice_number, s.created_at as sale_date, ct.amount,
                   -- Credit payments from credit_payments table (only actual payments toward credit debt)
                   COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
                   COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
                   ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
                   COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance
            FROM credit_transactions ct
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.customer_id = ?
            AND EXISTS (
                SELECT 1 FROM sale_items si 
                WHERE si.sale_id = s.id 
                AND si.id NOT IN (
                    SELECT DISTINCT ri.sale_item_id 
                    FROM refund_items ri 
                    JOIN refunds r ON ri.refund_id = r.id 
                    WHERE r.sale_id = s.id
                )
            )
            GROUP BY ct.id, ct.amount
            HAVING balance > 0
            ORDER BY s.created_at ASC
        ");
    $stmt->execute([$customerId]);
    $invoices = [];
    while ($row = $stmt->fetch()) {
        $invoices[] = [
            'id' => $row['id'],
            'invoice_number' => $row['invoice_number'],
            'sale_date' => $row['sale_date'],
            'credit_amount' => $row['amount'],
            'paid_amount' => $row['paid'],
            'waived_amount' => $row['waived'],
            'balance_raw' => $row['balance'], // Keep raw numeric value for calculations
            'balance_formatted' => number_format($row['balance'], 2) // Format for display
        ];
    }
    if (empty($invoices)) {
        echo json_encode([
            'success' => true,
            'invoices' => [],
            'debug' => [
                'customer_id' => $customerId,
                'sql' => 'SELECT ct.id, s.invoice_number, s.created_at as sale_date, ct.amount, (SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ct.id AND payment_method != \'refund\') as paid FROM credit_transactions ct JOIN sales s ON ct.sale_id = s.id WHERE ct.customer_id = ? GROUP BY ct.id HAVING ct.amount - paid > 0 ORDER BY s.created_at ASC',
                'message' => 'No unpaid invoices found for this customer.'
            ]
        ]);
        exit;
    } else {
        echo json_encode([
            'success' => true,
            'invoices' => $invoices,
            'debug' => [
                'customer_id' => $customerId,
                'count' => count($invoices),
                'message' => 'Unpaid invoices found.'
            ]
        ]);
        exit;
    }
}

// ... existing code ...
// Handle unpaid invoices request
if (isset($_GET['get_unpaid_invoices']) && isset($_GET['customer_id'])) {
    $customerId = $_GET['customer_id'];
    
    try {
        // Get unpaid invoices for the customer with waiver information (credit payments only)
        $stmt = $pdo->prepare("
            SELECT 
                ct.id,
                s.invoice_number,
                s.created_at as sale_date,
                ct.amount,
                -- Credit payments from credit_payments table (only actual payments toward credit debt, excluding refunds)
                COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) as paid,
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
                -- Subtract refunds from the original amount
                COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as refunded,
                ct.amount - COALESCE(SUM(CASE WHEN (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != 'refund' THEN cp.amount ELSE 0 END), 0) - 
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) - 
                COALESCE(SUM(CASE WHEN cp.payment_method = 'refund' THEN cp.amount ELSE 0 END), 0) as balance
            FROM credit_transactions ct
            JOIN sales s ON ct.sale_id = s.id
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.customer_id = ?
            GROUP BY ct.id, ct.amount
            HAVING balance > 0
            ORDER BY s.created_at DESC
        ");
        $stmt->execute([$customerId]);
        $invoices = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Keep balance as numeric for JavaScript calculations, format separately for display
        foreach ($invoices as &$invoice) {
            $invoice['credit_amount'] = $invoice['amount'];
            $invoice['paid_amount'] = $invoice['paid'];
            $invoice['waived_amount'] = $invoice['waived'];
            $invoice['balance_raw'] = $invoice['balance']; // Keep raw numeric value
            $invoice['balance_formatted'] = number_format($invoice['balance'], 2); // Format for display
        }
        
        header('Content-Type: application/json');
        echo json_encode(['success' => true, 'invoices' => $invoices]);
        exit;
    } catch (Exception $e) {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'message' => 'Error fetching unpaid invoices: ' . $e->getMessage()]);
        exit;
    }
}

// ... existing code ...

// Fetch all products for the add-credit-item modal with category and serial number info
$products = $pdo->query("
    SELECT p.id, p.name, p.price, p.cost_price, c.require_serials, c.name as category_name
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    ORDER BY p.name
")->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_credit_customer_id'])) {
    header('Content-Type: application/json');
    try {
        $customerId = intval($_POST['add_credit_customer_id']);
        $items = $_POST['items'] ?? [];
        $reminderTypes = $_POST['reminder_type'] ?? [];
        $userId = $_SESSION['user_id'];
        
        if ($customerId < 1 || empty($items)) {
            throw new Exception('Invalid input. Please select at least one product.');
        }
        
        // Validate all items and check stock availability
        $totalAmount = 0;
        $stockErrors = [];
        $priceErrors = [];
        
        foreach ($items as $item) {
            $productId = intval($item['product_id']);
            $quantity = intval($item['quantity']);
            $price = floatval($item['price']);
            
            if ($productId < 1 || $quantity < 1 || $price < 0) {
                throw new Exception('Invalid item data.');
            }
            
            // Check stock availability and cost price
            $stockStmt = $pdo->prepare("SELECT name, quantity, cost_price FROM products WHERE id = ?");
            $stockStmt->execute([$productId]);
            $product = $stockStmt->fetch();
            
            if (!$product) {
                throw new Exception('Product not found.');
            }
            
            if ($product['quantity'] < $quantity) {
                $stockErrors[] = "Insufficient stock for {$product['name']}. Available: {$product['quantity']}, Requested: {$quantity}";
            }
            
            // Check if price is below cost price
            if ($product['cost_price'] > 0 && $price < $product['cost_price']) {
                $priceErrors[] = "Price for {$product['name']} cannot be less than cost price ({$currency} " . number_format($product['cost_price'], 2) . "). Current price: {$currency} " . number_format($price, 2);
            }
            
            $totalAmount += $quantity * $price;
        }
        
        // If there are stock errors, throw exception
        if (!empty($stockErrors)) {
            throw new Exception('Stock validation failed: ' . implode('; ', $stockErrors));
        }
        
        // If there are price errors, throw exception
        if (!empty($priceErrors)) {
            throw new Exception('Price validation failed: ' . implode('; ', $priceErrors));
        }
        
        // Get customer details for messaging
        $stmt = $pdo->prepare("SELECT name, phone, email FROM customers WHERE id = ?");
        $stmt->execute([$customerId]);
        $customer = $stmt->fetch();
        
        // Use current balance data from Credit Transactions table instead of recalculating
        $previousTotalCredit = floatval($_POST['current_total_credit'] ?? 0);
        $totalPaidAmount = floatval($_POST['current_total_paid'] ?? 0);
        $totalWaivedAmount = floatval($_POST['current_total_waived'] ?? 0);
        $currentBalance = floatval($_POST['current_total_balance'] ?? 0);
        
        // Create balance info array for consistency
        $balanceInfo = [
            'total_credit' => $previousTotalCredit,
            'total_paid' => $totalPaidAmount,
            'total_waived' => $totalWaivedAmount
        ];
        
        $pdo->beginTransaction();
        
        // Generate unique invoice number
        $stmt = $pdo->query("SELECT MAX(CAST(SUBSTRING(invoice_number, 4) AS UNSIGNED)) as max_num FROM sales WHERE invoice_number LIKE 'INV%'");
        $result = $stmt->fetch();
        $nextNum = ($result['max_num'] ?? 0) + 1;
        $invoiceNumber = 'INV' . str_pad($nextNum, 6, '0', STR_PAD_LEFT);
        
        // Insert sale
        $stmt = $pdo->prepare("INSERT INTO sales (invoice_number, customer_id, user_id, total_amount, paid_amount, payment_method, status, payment_status, created_at, updated_at) VALUES (?, ?, ?, ?, 0, 'credit', 'pending', 'pending', NOW(), NOW())");
        $stmt->execute([$invoiceNumber, $customerId, $userId, $totalAmount]);
        $saleId = $pdo->lastInsertId();
        
        // Insert sale items
        $stmt = $pdo->prepare("INSERT INTO sale_items (sale_id, product_id, quantity, price, total, discount, unit_price, total_price) VALUES (?, ?, ?, ?, ?, 0, ?, ?)");
        
        foreach ($items as $item) {
            $productId = intval($item['product_id']);
            $quantity = intval($item['quantity']);
            $price = floatval($item['price']);
            $total = $quantity * $price;
            
            $stmt->execute([$saleId, $productId, $quantity, $price, $total, $price, $total]);
            
            // Deduct stock from product
            $stockStmt = $pdo->prepare("UPDATE products SET quantity = quantity - ? WHERE id = ?");
            $stockStmt->execute([$quantity, $productId]);
            error_log("Deducted {$quantity} units from product {$productId} stock for credit sale {$saleId}");
            
            // Handle serial numbers if provided
            if (!empty($item['serial_numbers'])) {
                try {
                    $serialNumbers = json_decode($item['serial_numbers'], true);
                    if (is_array($serialNumbers) && !empty($serialNumbers)) {
                        // Mark selected serial numbers as 'sold' and link to this credit sale
                        $serialStmt = $pdo->prepare("
                            UPDATE product_serial_numbers 
                            SET status = 'sold', source_type = 'credit_sale', source_id = ?, updated_at = NOW()
                            WHERE product_id = ? AND serial_number IN (" . str_repeat('?,', count($serialNumbers) - 1) . "?) AND status = 'available'
                        ");
                        
                        $params = array_merge([$saleId, $productId], $serialNumbers);
                        $result = $serialStmt->execute($params);
                        $updatedRows = $serialStmt->rowCount();
                        
                        error_log("Updated {$updatedRows} serial numbers to 'sold' status for product {$productId} via credit sale {$saleId}");
                        
                        // Verify all serial numbers were updated
                        if ($updatedRows !== count($serialNumbers)) {
                            error_log("Warning: Only {$updatedRows} out of " . count($serialNumbers) . " serial numbers were updated for product {$productId}");
                        }
                    }
                } catch (Exception $e) {
                    error_log("Error processing serial numbers for credit item: " . $e->getMessage());
                }
            }
        }
        
        // Insert credit transaction
        $stmt = $pdo->prepare("INSERT INTO credit_transactions (customer_id, sale_id, amount, type, status, due_date, created_at) VALUES (?, ?, ?, 'credit', 'pending', DATE_ADD(NOW(), INTERVAL 30 DAY), NOW())");
        $stmt->execute([$customerId, $saleId, $totalAmount]);
        
        // Set balance values for messages - use simple approach
        $previousTotalCredit = $balanceInfo['total_credit'];  // Total credit before adding new items
        $totalPaid = $balanceInfo['total_paid'];
        $totalWaived = $balanceInfo['total_waived'];          // Total waived amount
        $previousBalance = max(0, $previousTotalCredit - $totalPaid - $totalWaived); // Calculate previous balance (ensure non-negative)
        $totalCredit = $previousTotalCredit + $totalAmount;        // Total credit after adding new items
        $yourCurrentBalance = $totalCredit - $totalPaid - $totalWaived;     // Total Credit - Total Paid - Total Waived
        $totalOutstandingDebt = $yourCurrentBalance;              // Same as Your Current Balance
        
        // Debug logging for balance calculations (using current table values)
        error_log("=== Credit Items Added - Balance Debug (Using Current Table Values) ===");
        error_log("Customer ID: " . $customerId);
        error_log("Customer Name: " . $customer['name']);
        error_log("Source: Credit Transactions table data (not recalculated)");
        error_log("Raw balanceInfo: " . json_encode($balanceInfo));
        error_log("Previous Total Credit (from table): " . $previousTotalCredit);
        error_log("Total Paid (from table): " . $totalPaid);
        error_log("Total Waived (from table): " . $totalWaived);
        error_log("Current Balance (from table): " . $currentBalance);
        error_log("Previous Balance (Credit - Paid - Waived): " . $previousBalance);
        error_log("New Items Amount: " . $totalAmount);
        error_log("Total Credit (after adding new items): " . $totalCredit);
        error_log("Your Current Balance (Total Credit - Paid - Waived): " . $yourCurrentBalance);
        error_log("Total Outstanding Debt (same as Current Balance): " . $totalOutstandingDebt);
        error_log("=== End Balance Debug ===");
        
        // Prepare message content
        $itemDetails = [];
        foreach ($items as $item) {
            $productId = intval($item['product_id']);
            $quantity = intval($item['quantity']);
            $price = floatval($item['price']);
            $total = $quantity * $price;
            
            // Get product name
            $stmt = $pdo->prepare("SELECT name FROM products WHERE id = ?");
            $stmt->execute([$productId]);
            $productName = $stmt->fetchColumn();
            
            $itemDetails[] = $productName . " (Qty: " . $quantity . " @ " . $currency . " " . number_format($price, 2) . ")";
        }
        
        $itemsList = implode(", ", $itemDetails);
        
        // Include email templates
        require_once '../includes/email_templates.php';
        
        // Prepare items array for the email template
        $emailItems = [];
        foreach ($items as $item) {
            $stmt = $pdo->prepare("SELECT name FROM products WHERE id = ?");
            $stmt->execute([$item['product_id']]);
            $productName = $stmt->fetchColumn();
            
            $emailItem = [
                'name' => $productName,
                'quantity' => $item['quantity'],
                'price' => $item['price']
            ];
            
            // Add serial numbers if provided
            if (!empty($item['serial_numbers'])) {
                try {
                    $serialNumbers = json_decode($item['serial_numbers'], true);
                    if (is_array($serialNumbers) && !empty($serialNumbers)) {
                        $emailItem['serial_numbers'] = $serialNumbers;
                    }
                } catch (Exception $e) {
                    // Ignore JSON parse errors
                }
            }
            
            $emailItems[] = $emailItem;
        }
        
        // Create beautiful HTML email for email notifications
        $emailMessage = createCreditItemsAddedEmail(
            $customer['name'],
            $emailItems,
            $totalAmount,
            $yourCurrentBalance,
            $currency,
            $storeName,
            $storeAddress,
            $storePhone,
            $storeEmail,
            $storeWebsite,
            $customerId,
            $totalCredit,
            $totalPaid,
            $totalWaived
        );
        
        // Create detailed text message for WhatsApp/SMS (similar to reminder format)
        $textMessage = "Dear {$customer['name']},\n\n";
        $textMessage .= "New credit items have been added to your account at {$storeName}.\n\n";
        $textMessage .= "📋 CREDIT ITEMS ADDED:\n";
        $textMessage .= str_repeat('═', 40) . "\n";
        $textMessage .= "\n🧾 Invoice: {$invoiceNumber}\n";
        $textMessage .= "📅 Date: " . date('Y-m-d H:i') . "\n";
        $textMessage .= "📦 Items:\n";
        
        foreach ($items as $index => $item) {
            $stmt = $pdo->prepare("SELECT name FROM products WHERE id = ?");
            $stmt->execute([$item['product_id']]);
            $productName = $stmt->fetchColumn();
            
            $itemTotal = $item['quantity'] * $item['price'];
            $textMessage .= "   " . ($index + 1) . ". {$productName}\n";
            $textMessage .= "      Qty: {$item['quantity']} × {$currency} " . number_format($item['price'], 2) . " = {$currency} " . number_format($itemTotal, 2) . "\n";
            
            // Add serial numbers if provided
            if (!empty($item['serial_numbers'])) {
                try {
                    $serialNumbers = json_decode($item['serial_numbers'], true);
                    if (is_array($serialNumbers) && !empty($serialNumbers)) {
                        $textMessage .= "      Serial Numbers: " . implode(', ', $serialNumbers) . "\n";
                    }
                } catch (Exception $e) {
                    // Ignore JSON parse errors
                }
            }
        }
        
        $textMessage .= "📊 Previous Balance: {$currency} " . number_format($previousBalance, 2) . "\n";
        $textMessage .= "💰 Total Added: {$currency} " . number_format($totalAmount, 2) . "\n";
        $textMessage .= "💸 Waived: {$currency} " . number_format($totalWaived, 2) . "\n";
        $textMessage .= "⚠️ Your Current Balance: {$currency} " . number_format($yourCurrentBalance, 2) . "\n";
        $textMessage .= str_repeat('═', 40) . "\n\n";
        $textMessage .= "💳 Total Outstanding Debt: {$currency} " . number_format($totalOutstandingDebt, 2) . "\n\n";
        $textMessage .= "📞 Contact us on {$storePhone} for payment methods.\n";
        $textMessage .= "📍 Visit us at: {$storeAddress}\n\n";
        $textMessage .= "Thank you for your business!\n{$storeName}";
        
        // Debug the complete message content being sent
        error_log("=== Message Content Debug ===");
        error_log("Complete message being sent: " . $textMessage);
        error_log("=== End Message Content Debug ===");
        
        $subject = "Credit Items Added - {$storeName}";
        
        // Send messages based on selected reminder types
        $sentMessages = [];
        
        // If no reminder types are selected, skip messaging
        if (empty($reminderTypes)) {
            $successMessage = 'Credit items added successfully. No notifications sent (no reminder types selected).';
        } else {
            // Check SMS settings and send SMS (use text message)
            if (in_array('sms', $reminderTypes)) {
                $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'sms_settings'");
                $stmt->execute();
                $smsSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
                if (($smsSettings['enable_sms'] ?? '0') === '1' && !empty($customer['phone']) && 
                    !empty($smsSettings['sms_api_key']) && !empty($smsSettings['sms_api_secret']) && !empty($smsSettings['sms_sender_id'])) {
                    try {
                        $result = sendSMS($customer['phone'], $textMessage);
                        $sentMessages[] = 'SMS';
                    } catch (Exception $e) {
                        // SMS failed, but continue with other methods
                    }
                }
            }
            // Check Email settings and send Email (use HTML email)
            if (in_array('email', $reminderTypes)) {
                $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'email_settings'");
                $stmt->execute();
                $emailSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
                if (($emailSettings['enable_email'] ?? '0') === '1' && !empty($customer['email']) && 
                    !empty($emailSettings['smtp_host']) && !empty($emailSettings['smtp_port']) && 
                    !empty($emailSettings['smtp_username']) && !empty($emailSettings['smtp_password'])) {
                    try {
                        $result = sendEmail($customer['email'], $subject, $emailMessage);
                        $sentMessages[] = 'Email';
                        error_log("Credit items email sent successfully to: " . $customer['email']);
                    } catch (Exception $e) {
                        // Email failed, but continue with other methods
                        error_log("Credit items email failed for " . $customer['email'] . ": " . $e->getMessage());
                    }
                }
            }
            // Check WhatsApp settings and send WhatsApp (use text message)
            $whatsappUrl = null;
            if (in_array('whatsapp', $reminderTypes)) {
                $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'whatsapp_settings'");
                $stmt->execute();
                $whatsappSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
                if (($whatsappSettings['enable_whatsapp'] ?? '0') === '1' && !empty($customer['phone'])) {
                    try {
                        $result = sendWhatsApp($customer['phone'], $textMessage);
                        if ($result['success']) {
                            $sentMessages[] = 'WhatsApp';
                            error_log("WhatsApp link generated successfully for: " . $customer['phone']);
                            // Store WhatsApp URL for frontend handling
                            if (isset($result['whatsapp_url'])) {
                                $whatsappUrl = $result['whatsapp_url'];
                            }
                        } else {
                            error_log("WhatsApp generation failed for " . $customer['phone'] . ": " . ($result['message'] ?? 'Unknown error'));
                        }
                    } catch (Exception $e) {
                        // WhatsApp failed, but continue with other methods
                        error_log("WhatsApp error for " . $customer['phone'] . ": " . $e->getMessage());
                    }
                }
            }
            
            // Build detailed success message with message status
            $successMessage = 'Credit items added successfully.';
            $messageDetails = [];
            $failedMessages = [];
            
            // Check what was attempted and what succeeded
            if (in_array('sms', $reminderTypes)) {
                if (in_array('SMS', $sentMessages)) {
                    $messageDetails[] = 'SMS sent successfully';
                } else {
                    $failedMessages[] = 'SMS (not configured or failed)';
                }
            }
            
            if (in_array('email', $reminderTypes)) {
                if (in_array('Email', $sentMessages)) {
                    $messageDetails[] = 'Email sent successfully';
                } else {
                    $failedMessages[] = 'Email (not configured or failed)';
                }
            }
            
            if (in_array('whatsapp', $reminderTypes)) {
                if (in_array('WhatsApp', $sentMessages)) {
                    $messageDetails[] = 'WhatsApp sent successfully';
                } else {
                    $failedMessages[] = 'WhatsApp (not configured or failed)';
                }
            }
            
            if (!empty($messageDetails)) {
                $successMessage .= ' ' . implode(', ', $messageDetails) . '.';
            }
            
            if (!empty($failedMessages)) {
                $successMessage .= ' Failed: ' . implode(', ', $failedMessages) . '.';
            }
            
            if (empty($messageDetails) && empty($failedMessages)) {
                $successMessage .= ' No notifications sent (no reminder types selected).';
            }
        }
        
        // Log the messaging attempt
        $logDetails = json_encode([
            'customer_id' => $customerId,
            'customer_name' => $customer['name'],
            'invoice_number' => $invoiceNumber,
            'items_added' => $items,
            'total_amount' => $totalAmount,
            'source' => 'Credit Transactions table data (not recalculated)',
            'previous_total_credit_from_table' => $previousTotalCredit,
            'total_paid_from_table' => $totalPaid,
            'total_waived_from_table' => $totalWaived,
            'current_balance_from_table' => $currentBalance,
            'previous_balance' => $previousBalance,
            'total_credit' => $totalCredit,
            'your_current_balance' => $yourCurrentBalance,
            'total_outstanding_debt' => $totalOutstandingDebt,
            'messages_sent' => $sentMessages
        ]);
        
        $stmt = $pdo->prepare("
            INSERT INTO system_logs (user_id, action, details) 
            VALUES (?, 'credit_items_added', ?)
        ");
        $stmt->execute([$userId, $logDetails]);
        
        $pdo->commit();
        
        $response = ['success' => true, 'message' => $successMessage];
        
        // Include WhatsApp URL if available
        if ($whatsappUrl) {
            $response['whatsapp_url'] = $whatsappUrl;
            $response['customer_name'] = $customer['name'];
        }
        
        echo json_encode($response);
    } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

// Fetch WhatsApp and SMS settings
$settings = [];
$stmt = $pdo->query("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('enable_sms', 'enable_email', 'enable_whatsapp')");
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Credit Sales - EDDUH COMPUTERS</title>
    <?php
    // Get favicon path from database
    $stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'store_favicon' AND setting_group = 'store_info' LIMIT 1");
    $stmt->execute();
    $faviconPath = $stmt->fetchColumn();
    
    if ($faviconPath && file_exists('../' . $faviconPath)) {
        echo '<link rel="icon" type="image/x-icon" href="../' . htmlspecialchars($faviconPath) . '">';
        echo '<link rel="shortcut icon" type="image/x-icon" href="../' . htmlspecialchars($faviconPath) . '">';
    } else {
        echo '<link rel="icon" type="image/x-icon" href="../assets/images/favicon.ico">';
        echo '<link rel="shortcut icon" type="image/x-icon" href="../assets/images/favicon.ico">';
    }
    ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <style>
        body { 
            background: #f8fafc; 
            font-family: 'Inter', sans-serif; 
        }
        .card { 
            border-radius: 16px; 
            box-shadow: 0 4px 20px rgba(0,0,0,0.05); 
            border: none; 
            transition: transform 0.2s;
        }
        .card:hover {
            transform: translateY(-5px);
        }
        .card-header { 
            background: none; 
            border-bottom: none; 
            padding: 1.5rem 1.5rem 0.5rem;
        }
        .card-body {
            padding: 1.5rem;
        }
        .stat-card {
            background: linear-gradient(135deg, #fff 0%, #f8fafc 100%);
            border-left: 4px solid;
        }
        .stat-card.primary { border-left-color: #3b82f6; }
        .stat-card.success { border-left-color: #10b981; }
        .stat-card.info { border-left-color: #06b6d4; }
        .stat-card.warning { border-left-color: #f59e0b; }
        .stat-value {
            font-size: 1.8rem;
            font-weight: 700;
            color: #1e293b;
            margin: 0.5rem 0;
        }
        .stat-label {
            color: #64748b;
            font-size: 0.9rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        .table {
            background: #fff;
            border-radius: 12px;
            overflow: hidden;
        }
        .table th {
            background: #f8fafc;
            font-weight: 600;
            color: #475569;
            border-bottom: 2px solid #e2e8f0;
        }
        .table td {
            vertical-align: middle;
            color: #334155;
        }
        .btn {
            border-radius: 8px;
            font-weight: 500;
            padding: 0.5rem 1rem;
        }
        .btn-primary {
            background: #3b82f6;
            border: none;
        }
        .btn-primary:hover {
            background: #2563eb;
        }
        .filter-card {
            background: #fff;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 1rem;
        }
        .filter-card .form-control {
            border-radius: 6px;
            border: 1px solid #e2e8f0;
        }
        .filter-card .form-control:focus {
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        .badge {
            padding: 0.5em 1em;
            border-radius: 6px;
            font-weight: 500;
        }
        .badge-paid {
            background: #10b981;
            color: white;
        }
        .badge-pending {
            background: #f59e0b;
            color: white;
        }
        .badge-warning, .badge-partially-paid {
            background: #fbbf24;
            color: #111;
        }
        .progress {
            height: 8px;
            border-radius: 4px;
            background: #e2e8f0;
        }
        .progress-bar {
            border-radius: 4px;
        }
        /* Credit Table Column Widths */
        .credit-table th:nth-child(1), .credit-table td:nth-child(1) { /* Date */
            width: 10%;
            min-width: 80px;
        }
        .credit-table th:nth-child(2), .credit-table td:nth-child(2) { /* Invoice */
            width: 12%;
            min-width: 100px;
        }
        .credit-table th:nth-child(3), .credit-table td:nth-child(3) { /* Items */
            width: 20%;
            min-width: 150px;
        }
        .credit-table th:nth-child(4), .credit-table td:nth-child(4) { /* Qty */
            width: 6%;
            min-width: 50px;
            text-align: center;
        }
        .credit-table th:nth-child(5), .credit-table td:nth-child(5) { /* Payment Details */
            width: 18%;
            min-width: 140px;
        }
        .credit-table th:nth-child(6), .credit-table td:nth-child(6) { /* Credit Amount */
            width: 12%;
            min-width: 100px;
            text-align: right;
        }
        .credit-table th:nth-child(7), .credit-table td:nth-child(7) { /* Paid */
            width: 10%;
            min-width: 80px;
            text-align: right;
        }
        .credit-table th:nth-child(8), .credit-table td:nth-child(8) { /* Balance */
            width: 12%;
            min-width: 100px;
            text-align: right;
        }
        .credit-table th:nth-child(9), .credit-table td:nth-child(9) { /* Status */
            width: 10%;
            min-width: 120px;
            text-align: center;
        }
        /* Additional table styling */
        .credit-table {
            font-size: 0.875rem;
        }
        .credit-table th {
            font-weight: 600;
            background: #f8fafc;
            border-bottom: 2px solid #e2e8f0;
            white-space: nowrap;
        }
        .credit-table td {
            border-bottom: 1px solid #e2e8f0;
            padding: 0.5rem 0.25rem;
        }
        .credit-table td:nth-child(3) { /* Items column */
            max-width: 0;
            word-wrap: break-word;
            white-space: normal;
            overflow-wrap: break-word;
            hyphens: auto;
        }
        .credit-table td:nth-child(5) { /* Payment Details column */
            max-width: 0;
            word-wrap: break-word;
            white-space: normal;
            overflow-wrap: break-word;
            hyphens: auto;
        }
        /* Print Report Styles */
        @media print {
            body *:not(.print-section):not(.print-section *) {
                display: none !important;
            }
            .print-section {
                display: block !important;
                width: 100%;
                margin: 0;
                padding: 0;
            }
            .print-footer {
                position: fixed;
                bottom: 0;
                left: 0;
                width: 100%;
                text-align: center;
                font-size: 0.95rem;
                color: #444;
                border-top: 1px solid #e5e7eb;
                background: #fff;
                padding: 8px 0 4px 0;
            }
            .print-footer .page-number:after {
                content: "Page " counter(page) " of " counter(pages);
            }
        }
        /* Dropdown styles */
        .dropdown-menu {
            min-width: 250px;
            padding: 0.5rem 0;
            margin: 0;
            border: 1px solid rgba(0,0,0,.15);
            border-radius: 0.25rem;
            box-shadow: 0 0.5rem 1rem rgba(0,0,0,.175);
            z-index: 9999 !important;
            position: absolute !important;
        }
        .dropdown-item {
            padding: 0.5rem 1rem;
            clear: both;
            font-weight: 400;
            color: #212529;
            text-align: inherit;
            white-space: nowrap;
            background-color: transparent;
            border: 0;
            cursor: pointer;
        }
        .dropdown-item:hover {
            color: #1e2125;
            background-color: #e9ecef;
        }
        .dropdown-item-form {
            margin: 0;
            padding: 0;
        }
        .dropdown-item-form button {
            width: 100%;
            text-align: left;
            padding: 0.5rem 1rem;
            border: none;
            background: none;
        }
        .dropdown-item-form button:hover {
            background-color: #e9ecef;
        }
        .btn-group {
            position: relative;
        }
        /* Add styles for the reminder dropdown */
        .reminder-dropdown {
            position: relative;
        }
        .reminder-dropdown .dropdown-menu {
            transform: translate3d(0px, 38px, 0px) !important;
            top: 0 !important;
            left: 0 !important;
            will-change: transform;
        }
        /* Ensure the dropdown stays above other content */
        .table-responsive {
            position: relative;
        }
        .table-responsive .dropdown-menu {
            position: absolute !important;
        }
    </style>
</head>
<body>
<div class="container-fluid py-4">
    <div class="row">
        <?php include '../includes/sidebar.php'; ?>
        <?php include '../includes/topbar.php'; ?>
        <!-- Main Content -->
        <div class="col-md-10 ms-sm-auto px-md-4">
            <div class="container-fluid mt-4">
                <!-- Filters and Header (single card) -->
                <div class="card mb-4 no-print-filters">
                    <div class="card-body py-2">
                        <div class="d-flex justify-content-between align-items-center mb-2">
                            <h2 class="mb-0 fw-bold">Credit Sales Management</h2>
                    <a href="Cleared Credit Sales.php" target="_blank" class="btn btn-outline-success" id="showClearedCreditsBtn">
                        <i class="fas fa-check-circle"></i> Cleared Credit Sales
                    </a>
                        </div>
                        <form method="GET" class="row gx-2 gy-1 align-items-end flex-nowrap">
                            <div class="col-md-2">
                                <label class="form-label mb-1">Status</label>
                                <select name="status" class="form-select form-select-lg">
                                    <option value="">All Status</option>
                                    <option value="paid" <?php echo $status === 'paid' ? 'selected' : ''; ?>>Paid</option>
                                    <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label mb-1">Customer</label>
                                <select name="customer_id" class="form-select form-select-lg">
                                    <option value="">All Customers</option>
                                    <?php foreach ($customers as $customer): ?>
                                    <option value="<?php echo $customer['id']; ?>" <?php echo $customer_id == $customer['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($customer['name']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label class="form-label mb-1">Search</label>
                                <input type="text" name="search" class="form-control form-control-lg" placeholder="Search by invoice or customer..." value="<?php echo htmlspecialchars($search); ?>">
                            </div>
                            <div class="col-md-2 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary btn-lg w-100">Apply</button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Statistics Cards -->
                <div class="row mb-4 no-print-stats">
                    <div class="col-md-3">
                        <div class="card stat-card primary">
                            <div class="card-body">
                                <div class="stat-label">Total Credit</div>
                                <div class="stat-value"><?php echo $currency . ' ' . number_format($total_credit, 2); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-primary" style="width: 100%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card success">
                            <div class="card-body">
                                <div class="stat-label">Amount Paid</div>
                                <div class="stat-value"><?php echo $currency . ' ' . number_format($total_paid, 2); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-success" style="width: <?php echo ($total_credit > 0) ? ($total_paid / $total_credit * 100) : 0; ?>%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card warning">
                            <div class="card-body">
                                <div class="stat-label">Pending Amount</div>
                                <div class="stat-value"><?php echo $currency . ' ' . number_format($total_pending, 2); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-warning" style="width: <?php echo ($total_credit > 0) ? ($total_pending / $total_credit * 100) : 0; ?>%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card info">
                            <div class="card-body">
                                <div class="stat-label">Credit Customers</div>
                                <div class="stat-value"><?php echo number_format($total_customers); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-info" style="width: 100%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Cleared Credit Sales Table (hidden by default) -->
                <div id="clearedCreditsSection" style="display:none; margin-bottom:2rem;">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">Cleared Credit Sales</h5>
                        </div>
                        <div class="card-body p-0">
                            <div class="table-responsive">
                                <table class="table table-hover mb-0" id="clearedCreditsTable">
                                    <thead>
                                        <tr>
                                            <th>Customer</th>
                                            <th>Total Credit</th>
                                            <th>Total Paid</th>
                                            <th>Total Balance</th>
                                            <th>Credits</th>
                                            <th>Print</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($cleared_credits_by_customer as $cid => $group): ?>
                                        <tr class="cleared-customer-row" data-customer-id="<?php echo $cid; ?>">
                                            <td>
                                                <a href="#" class="toggle-cleared-transactions" data-customer-id="<?php echo $cid; ?>" style="text-decoration:underline; cursor:pointer;">
                                                    <?php echo htmlspecialchars($group['customer_name']); ?>
                                                </a>
                                            </td>
                                            <td><?php echo $currency . ' ' . number_format($group['total_credit'], 2); ?></td>
                                            <td><?php echo $currency . ' ' . number_format($group['total_paid'], 2); ?></td>
                                            <td><?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?></td>
                                            <td><span class="badge bg-info"><?php echo count($group['transactions']); ?> credits</span></td>
                                            <td><button class="btn btn-sm btn-outline-primary print-cleared-receipt" data-customer-id="<?php echo $cid; ?>"><i class="fas fa-print"></i> Print</button></td>
                                        </tr>
                                        <?php foreach ($group['transactions'] as $credit): ?>
                                        <?php if ($credit['status'] !== 'paid') continue; ?>
                                        <tr class="cleared-transaction-row" data-customer-id="<?php echo $cid; ?>" style="display:none; background:#f8fafc;">
                                            <td colspan="1"></td>
                                            <td colspan="5">
                                                <div style="display:flex; flex-direction:column; gap:8px;">
                                                    <div style="display:flex; gap:24px; align-items:center;">
                                                        <span><b>Date:</b> <?php echo date('M d, Y', strtotime($credit['sale_date'])); ?></span>
                                                        <span><b>Invoice:</b> <?php echo htmlspecialchars($credit['invoice_number']); ?></span>
                                                        <span><b>Amount:</b> <?php echo $currency . ' ' . number_format($credit['amount'], 2); ?></span>
                                                        <span><b>Paid:</b> <?php echo $currency . ' ' . number_format($credit['amount_paid'], 2); ?></span>
                                                        <span><b>Status:</b> <?php echo ucfirst($credit['status']); ?></span>
                                                    </div>
                                                    <div style="padding-left:16px;">
                                                        <b>Items:</b> <?php echo htmlspecialchars($credit['items'] ?? 'No items found'); ?>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                        <!-- Hidden receipt template for this customer -->
                                        <template id="receipt-template-<?php echo $cid; ?>" data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" data-total-paid="<?php echo $group['total_paid']; ?>">
                                            <div style="width:80mm; font-family:monospace; font-size:13px; color:#222;">
                                                <div style="text-align:center; margin-bottom:8px;">
                                                    <div style="font-size:1.2rem; font-weight:700; color:#222;">POS</div>
                                                    <div><?php echo htmlspecialchars($storeAddress); ?></div>
                                                    <div>Phone: <?php echo htmlspecialchars($storePhone); ?></div>
                                                    <div>Email: <?php echo htmlspecialchars($storeEmail); ?></div>
                                                    <div>Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
                                                </div>
                                                <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                                <div><b>Customer:</b> <?php echo htmlspecialchars($group['customer_name']); ?></div>
                                                <div><b>Date:</b> <?php echo date('Y-m-d H:i:s'); ?></div>
                                                <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                                <div><b>Cleared Credit Items:</b></div>
                                                <table style="width:100%; font-size:12px; border-collapse:collapse;">
                                                    <thead>
                                                        <tr>
                                                            <th style="text-align:left;">Date</th>
                                                            <th style="text-align:left;">Invoice</th>
                                                            <th style="text-align:right;">Amount</th>
                                                            <th style="text-align:right;">Waived</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php foreach ($group['transactions'] as $item): ?>
                                                        <?php if ($item['status'] !== 'paid') continue; ?>
                                                        <?php
                                                        // Calculate total waived for this credit transaction
                                                        $waived = 0;
                                                        $waiveStmt = $pdo->prepare("SELECT SUM(amount) FROM credit_payments WHERE credit_transaction_id = ? AND is_waived = 1");
                                                        $waiveStmt->execute([$item['id']]);
                                                        $waived = $waiveStmt->fetchColumn();
                                                        ?>
                                                        <tr>
                                                            <td><?php echo date('M d', strtotime($item['sale_date'])); ?></td>
                                                            <td><?php echo htmlspecialchars($item['invoice_number']); ?></td>
                                                            <td style="text-align:right;"><?php echo $currency . ' ' . number_format($item['amount'], 2); ?></td>
                                                            <td style="text-align:right; color:#dc2626;"><?php echo $waived > 0 ? ($currency . ' ' . number_format($waived, 2)) : '-'; ?></td>
                                                        </tr>
                                                        <?php endforeach; ?>
                                                    </tbody>
                                                </table>
                                                <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                                <div><b>Total Paid:</b> <?php echo $currency . ' ' . number_format($group['total_paid'], 2); ?></div>
                                                <div><b>Total Waived:</b> <?php echo $currency . ' ' . number_format($group['total_waived'], 2); ?></div>
                                                <div style="margin:12px 0 4px 0; text-align:center;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
                                                <div style="text-align:center; margin:8px 0;">
                                                    <img id="receipt-qr" style="width:120px; height:120px;" />
                                                </div>
                                                <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                                <div style="text-align:center; font-size:11px;">Designed and developed by Vicotechnologies.co.ke<br>+254702214793</div>
                                            </div>
                                        </template>
                                        <?php endforeach; ?>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Credit Transactions Table -->
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="card-title mb-0">Credit Transactions</h5>
                        <span class="badge bg-primary">Grouped by Customer</span>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0" id="creditTransactionsTable">
                                <thead>
                                    <tr>
                                        <th>Customer</th>
                                        <th>Total Credit</th>
                                        <th>Total Paid</th>
                                        <th>Total Waived</th>
                                        <th>Total Balance</th>
                                        <th>Status</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($credits_by_customer as $cid => $group): ?>
                                    <?php
                                    // Filter out customers with only fully paid invoices
                                    $has_unpaid = false;
                                    foreach ($group['transactions'] as $credit) {
                                        $item_balance = $credit['amount'] - $credit['amount_paid'];
                                        if ($item_balance > 0.001) {
                                            $has_unpaid = true;
                                            break;
                                        }
                                    }
                                    if (!$has_unpaid) continue;
                                    ?>
                                    <tr class="customer-row" data-customer-id="<?php echo $cid; ?>">
                                        <td>
                                            <a href="#" class="customer-name" data-customer-id="<?php echo $cid; ?>" style="text-decoration:none; color:inherit; display:flex; align-items:center; gap:8px;">
                                                <i class="fas fa-chevron-right" style="font-size:0.8rem;"></i>
                                                <?php echo htmlspecialchars($group['customer_name']); ?>
                                            </a>
                                        </td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_credit'], 2); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_paid'], 2); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_waived'], 2); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?></td>
                                        <td>
                                            <?php
                                            $statusClass = '';
                                            switch($group['status']) {
                                                case 'Not Paid':
                                                    $statusClass = 'bg-danger';
                                                    break;
                                                case 'Partially Paid':
                                                    $statusClass = 'bg-warning text-dark';
                                                    break;
                                                default:
                                                    $statusClass = 'bg-danger';
                                            }
                                            ?>
                                            <span class="badge <?php echo $statusClass; ?>"><?php echo $group['status']; ?></span>
                                        </td>
                                        <td>
                                            <div class="btn-group">
                                                <button type="button" class="btn btn-sm btn-success pay-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" 
                                                    data-balance="<?php echo $group['total_balance']; ?>">
                                                    <i class="fas fa-money-bill"></i> Pay
                                                </button>
                                                <button type="button" class="btn btn-sm btn-warning waive-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" 
                                                    data-balance="<?php echo $group['total_balance']; ?>">
                                                    <i class="fas fa-hand-holding-heart"></i> Waive
                                                </button>
                                                <button type="button" class="btn btn-sm btn-info reminder-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" 
                                                    data-balance="<?php echo $group['total_balance']; ?>">
                                                    <i class="fas fa-bell"></i> Remind
                                                </button>
                                                <button type="button" class="btn btn-sm btn-primary print-credit-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>">
                                                    <i class="fas fa-print"></i> Print
                                                </button>
                                                <button type="button" class="btn btn-sm btn-dark add-credit-item-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>"
                                                    data-total-credit="<?php echo $group['total_credit']; ?>"
                                                    data-total-paid="<?php echo $group['total_paid']; ?>"
                                                    data-total-waived="<?php echo $group['total_waived']; ?>"
                                                    data-total-balance="<?php echo $group['total_balance']; ?>">
                                                    <i class="fas fa-plus"></i> Add
                                                </button>
                                                <button type="button" class="btn btn-sm btn-secondary edit-invoice-btn" 
                                                    data-customer-id="<?php echo $cid; ?>" 
                                                    data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>">
                                                    <i class="fas fa-edit"></i> Edit
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                    <tr class="credit-items" data-customer-id="<?php echo $cid; ?>" style="display:none; background:#f8fafc;">
                                        <td colspan="6">
                                            <div class="p-3">
                                                <h6 class="mb-3">Credit Items for <?php echo htmlspecialchars($group['customer_name']); ?></h6>
                                                <div class="table-responsive">
                                                    <table class="table table-sm credit-table">
                                                        <thead>
                                                            <tr style="background:#f1f5f9; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                                                                <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Date</th>
                                                                <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Invoice</th>
                                                                <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Items</th>
                                                                <th style="text-align:center; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Qty</th>
                                                                <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Payment Details</th>
                                                                <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Waived</th>
                                                                <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Credit Amount</th>
                                                                <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Paid</th>
                                                                <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Balance</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php $rowAlt = false; foreach ($group['transactions'] as $credit): ?>
                                                            <?php
                                                            // Calculate waived amount for this transaction
                                                            $waiveStmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ? AND is_waived = 1");
                                                            $waiveStmt->execute([$credit['id']]);
                                                            $waived = $waiveStmt->fetchColumn();
                                                            
                                                            // Calculate refunded amount for this transaction
                                                            $refundStmt = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM credit_payments WHERE credit_transaction_id = ? AND payment_method = 'refund'");
                                                            $refundStmt->execute([$credit['id']]);
                                                            $refunded = $refundStmt->fetchColumn();
                                                            
                                                            // Calculate adjusted credit amount (original minus refunded)
                                                            $adjusted_credit_amount = $credit['amount'] - $refunded;
                                                            
                                                            // Calculate balance using adjusted amount
                                                            $item_balance = $adjusted_credit_amount - $credit['amount_paid'] - $waived; 
                                                            if ($item_balance <= 0.001) continue; 
                                                            ?>
                                                            <tr style="background:<?php echo $rowAlt ? '#f8fafc' : '#fff'; ?>; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                                                                <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo date('M d', strtotime($credit['sale_date'])); ?></td>
                                                                <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($credit['invoice_number']); ?></td>
                                                                <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($credit['items'] ?? 'No items found'); ?></td>
                                                                <td style="text-align:center; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $credit['total_quantity'] ?? 0; ?></td>
                                                                <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                                                                    <?php
                                                                    $paymentDetails = '';
                                                                    
                                                                    // Show payment amounts minus returned items
                                                                    if (!empty($credit['payment_methods'])) {
                                                                        $methods = explode(', ', $credit['payment_methods']);
                                                                        $amounts = explode(', ', $credit['payment_amounts']);
                                                                        $details = [];
                                                                        for ($i = 0; $i < count($methods); $i++) {
                                                                            if (isset($amounts[$i]) && $amounts[$i] > 0) {
                                                                                $adjusted_amount = max(0, $amounts[$i] - $refunded);
                                                                                if ($adjusted_amount > 0) {
                                                                                    $details[] = ucfirst($methods[$i]) . ': ' . $currency . ' ' . number_format($adjusted_amount, 2);
                                                                                }
                                                                            }
                                                                        }
                                                                        $paymentDetails = !empty($details) ? implode(', ', $details) : 'Credit Sale';
                                                                    } elseif (!empty($credit['sale_payment_method']) && $credit['sale_paid_amount'] > 0) {
                                                                        $adjusted_amount = max(0, $credit['sale_paid_amount'] - $refunded);
                                                                        $paymentDetails = $adjusted_amount > 0 ? ucfirst($credit['sale_payment_method']) . ': ' . $currency . ' ' . number_format($adjusted_amount, 2) : 'Credit Sale';
                                                                    } else {
                                                                        $paymentDetails = 'Credit Sale';
                                                                    }
                                                                    
                                                                    echo $paymentDetails;
                                                                    ?>
                                                                </td>
                                                                <td style="text-align:right; padding:5px 4px; color:#dc2626; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $waived > 0 ? ($currency . ' ' . number_format($waived, 2)) : '-'; ?></td>
                                                                <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($adjusted_credit_amount, 2); ?></td>
                                                                <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount_paid'], 2); ?></td>
                                                                <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($item_balance, 2); ?></td>
                                                            </tr>
                                                            <?php $rowAlt = !$rowAlt; endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>


<!-- Payment Modal -->
<div class="modal fade" id="paymentModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Process Payment</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="paymentForm" method="POST">
                    <input type="hidden" name="payment_id" id="paymentId">
                    <input type="hidden" name="payment_customer_id" id="paymentCustomerId">
                    <div class="mb-3">
                        <label class="form-label">Customer</label>
                        <input type="text" class="form-control" id="customerName" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Select Unpaid Invoices</label>
                        <div id="paymentInvoiceList" style="max-height:150px; overflow-y:auto; border:1px solid #eee; border-radius:6px; padding:8px; background:#fafbfc;"></div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Amount</label>
                        <input type="number" class="form-control" name="amount" id="paymentAmount" step="0.01" min="1" required>
                        <div class="invalid-feedback">Minimum payment amount is 1</div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Payment Method</label>
                        <select class="form-select" name="payment_method" id="paymentMethod" required>
                            <option value="mpesa_received">Sent Via M-PESA(Received)</option>
                            <option value="cash">Cash</option>
                            <option value="mpesa">M-Pesa</option>
                        </select>
                    </div>
                    <div class="mb-3" id="mpesaPhoneGroup" style="display: none;">
                        <label class="form-label">M-Pesa Phone Number</label>
                        <input type="text" class="form-control" id="mpesaPhone" name="mpesa_phone" placeholder="e.g., 254712345678">
                        <div class="invalid-feedback">Please enter a valid M-Pesa phone number</div>
                    </div>
                    <div class="mb-3" id="referenceGroup" style="display: none;">
                        <label class="form-label">MPESA Transaction Code or Phone(Optional)</label>
                        <input type="text" class="form-control" name="reference" placeholder="Enter transaction code or phone number">
                    </div>
                    
                    <!-- Notification Options -->
                    <div class="mb-3">
                        <label class="form-label">
                            <i class="fas fa-bell me-2"></i>Send Payment Notifications
                        </label>
                        <div class="card bg-light">
                            <div class="card-body py-2">
                                <div class="row">
                                    <div class="col-md-4">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="send_sms_payment" name="send_sms" <?php echo ($settings['enable_sms'] ?? '0') === '1' ? 'checked' : ''; ?> <?php echo ($settings['enable_sms'] ?? '0') !== '1' ? 'disabled' : ''; ?>>
                                            <label class="form-check-label small" for="send_sms_payment">
                                                <i class="fas fa-sms text-primary me-1"></i>SMS
                                                <?php if (($settings['enable_sms'] ?? '0') !== '1'): ?>
                                                    <small class="text-muted">(Disabled)</small>
                                                <?php endif; ?>
                                            </label>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="send_email_payment" name="send_email" <?php echo ($settings['enable_email'] ?? '0') === '1' ? 'checked' : ''; ?> <?php echo ($settings['enable_email'] ?? '0') !== '1' ? 'disabled' : ''; ?>>
                                            <label class="form-check-label small" for="send_email_payment">
                                                <i class="fas fa-envelope text-info me-1"></i>Email
                                                <?php if (($settings['enable_email'] ?? '0') !== '1'): ?>
                                                    <small class="text-muted">(Disabled)</small>
                                                <?php endif; ?>
                                            </label>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="send_whatsapp_payment" name="send_whatsapp" <?php echo ($settings['enable_whatsapp'] ?? '0') === '1' ? 'checked' : ''; ?> <?php echo ($settings['enable_whatsapp'] ?? '0') !== '1' ? 'disabled' : ''; ?>>
                                            <label class="form-check-label small" for="send_whatsapp_payment">
                                                <i class="fab fa-whatsapp text-success me-1"></i>WhatsApp
                                                <?php if (($settings['enable_whatsapp'] ?? '0') !== '1'): ?>
                                                    <small class="text-muted">(Disabled)</small>
                                                <?php endif; ?>
                                            </label>
                                        </div>
                                    </div>
                                </div>
                                <div class="row mt-2">
                                    <div class="col-12">
                                        <button type="button" class="btn btn-outline-secondary btn-sm w-100" id="previewPaymentMessage">
                                            <i class="fas fa-eye me-1"></i>Preview Message
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="text-end">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary" data-permission="process_credit_payment" data-permission-label="Process Credit Payment">Process Payment</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Payment Message Preview Modal -->
<div class="modal fade" id="paymentMessagePreviewModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-eye me-2"></i>Payment Notification Preview
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <nav class="nav nav-tabs mb-3" id="previewTabs">
                    <a class="nav-link active" id="sms-tab" data-bs-toggle="tab" href="#sms-preview">
                        <i class="fas fa-sms me-1"></i>SMS/WhatsApp
                    </a>
                    <a class="nav-link" id="email-tab" data-bs-toggle="tab" href="#email-preview">
                        <i class="fas fa-envelope me-1"></i>Email
                    </a>
                </nav>
                
                <div class="tab-content">
                    <!-- SMS/WhatsApp Preview -->
                    <div class="tab-pane fade show active" id="sms-preview">
                        <div class="card">
                            <div class="card-header bg-primary text-white">
                                <h6 class="mb-0">
                                    <i class="fas fa-mobile-alt me-2"></i>SMS & WhatsApp Message
                                </h6>
                            </div>
                            <div class="card-body">
                                <div class="p-3 bg-light rounded border" style="font-family: monospace; white-space: pre-wrap;" id="smsPreviewContent">
                                    <!-- SMS content will be loaded here -->
                                </div>
                                <div class="mt-2">
                                    <small class="text-muted">
                                        <i class="fas fa-info-circle me-1"></i>
                                        This message will be sent via SMS and WhatsApp if enabled.
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Email Preview -->
                    <div class="tab-pane fade" id="email-preview">
                        <div class="card">
                            <div class="card-header bg-info text-white">
                                <h6 class="mb-0">
                                    <i class="fas fa-envelope me-2"></i>Email Message
                                </h6>
                            </div>
                            <div class="card-body">
                                <div class="mb-2">
                                    <strong>Subject:</strong> <span id="emailSubjectPreview">Payment Confirmation - Store Name</span>
                                </div>
                                <div class="border rounded p-3" style="max-height: 400px; overflow-y: auto;" id="emailPreviewContent">
                                    <!-- Email content will be loaded here -->
                                </div>
                                <div class="mt-2">
                                    <small class="text-muted">
                                        <i class="fas fa-info-circle me-1"></i>
                                        This is a preview of the HTML email that will be sent.
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Payment Receipt Template -->
<template id="payment-receipt-template-<?php echo $cid; ?>" data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" data-total-balance="<?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?>">
    <div style="width:80mm; font-family: Arial, Helvetica, sans-serif; font-size:13px; color:#111; font-weight:700;">
        <div style="text-align:center; margin-bottom:10px; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <div style="font-size:1.1rem; font-weight:900; color:#000; margin-bottom:2px;">PAYMENT RECEIPT</div>
            <?php if ($logoData): ?>
                <?php
                $mimeType = 'image/jpeg'; // Default to JPEG
                if (strpos(strtolower($logoName), '.png') !== false) {
                    $mimeType = 'image/png';
                } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                    $mimeType = 'image/gif';
                }
                ?>
                <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
            <?php endif; ?>
        </div>
        <div style="text-align:center; margin-bottom:10px; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <div style="font-size:1.1rem; font-weight:700; color:#111; font-family: Arial, Helvetica, sans-serif;"><?php echo htmlspecialchars($storeName); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($storeAddress); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Phone: <?php echo htmlspecialchars($storePhone); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Email: <?php echo htmlspecialchars($storeEmail); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
        </div>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Customer:</b> <?php echo htmlspecialchars($group['customer_name']); ?></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Date:</b> <span class="receipt-date"></span></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Receipt No:</b> <span class="receipt-number"></span></div>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-weight:700; margin-bottom:4px; font-family: Arial, Helvetica, sans-serif;">Payment Details:</div>
        <table style="width:100%; font-size:12px; border-collapse:collapse; margin-bottom:10px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <thead>
                <tr style="background:#f1f5f9; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Date</th>
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Invoice</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Amount</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Paid</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Balance</th>
                </tr>
            </thead>
            <tbody>
                <?php $rowAlt = false; foreach ($group['transactions'] as $credit): ?>
                <?php $item_balance = $credit['amount'] - $credit['amount_paid']; if ($item_balance <= 0.001) continue; ?>
                <tr style="background:<?php echo $rowAlt ? '#f8fafc' : '#fff'; ?>; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo date('M d', strtotime($credit['sale_date'])); ?></td>
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($credit['invoice_number']); ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount'], 2); ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount_paid'], 2); ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($item_balance, 2); ?></td>
                </tr>
                <?php $rowAlt = !$rowAlt; endforeach; ?>
            </tbody>
        </table>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Total Balance:</b> <?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?></div>
        <div style="margin:14px 0 6px 0; text-align:center; font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
        <div style="font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Total Paid:</b> <?php echo $currency . ' ' . number_format($group['total_paid'], 2); ?></div>
        <div style="font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Total Waived:</b> <?php echo $currency . ' ' . number_format($group['total_waived'], 2); ?></div>
    </div>
</template>

<!-- Add waive Modal -->
<div class="modal fade" id="waiveModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Waive Amount</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="waiveForm" method="POST">
                    <input type="hidden" name="waive_customer_id" id="waiveCustomerId">
                    <div class="mb-3">
                        <label class="form-label">Customer</label>
                        <input type="text" class="form-control" id="waiveCustomerName" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Select Unpaid Invoices</label>
                        <div id="waiveInvoiceList" style="max-height:150px; overflow-y:auto; border:1px solid #eee; border-radius:6px; padding:8px; background:#fafbfc;"></div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Amount to waive</label>
                        <input type="number" class="form-control" name="waive_amount" id="waiveAmount" step="0.01" min="0.01" required>
                        <div class="invalid-feedback">Waive amount must be greater than 0</div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Reason</label>
                        <textarea class="form-control" name="waive_reason" id="waiveReason" rows="3" required></textarea>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-warning" data-permission="waive_credit" data-permission-label="Waive Credit">Waive Amount</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Add waive Receipt Template -->
<template id="waive-receipt-template">
    <div style="width:80mm; font-family: 'Arial Black', Arial, sans-serif; font-size:13px; color:#111; font-weight:900; padding:0; margin:0;">
        <div style="width:100%; text-align:center; color:#000; font-weight:900; margin-bottom:2px; font-size:1.1rem;">WAIVER RECEIPT</div>
        <?php if ($logoData): ?>
            <?php
            $mimeType = 'image/jpeg'; // Default to JPEG
            if (strpos(strtolower($logoName), '.png') !== false) {
                $mimeType = 'image/png';
            } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                $mimeType = 'image/gif';
            }
            ?>
            <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
        <?php endif; ?>
        <div style="text-align:center; margin-bottom:8px; width:100%;">
            <div style="font-size:1.2rem; font-weight:700; color:#000;"><?php echo htmlspecialchars($storeName); ?></div>
            <div style="width:100%; color:#111; font-weight:600; "><?php echo htmlspecialchars($storeAddress); ?></div>
            <div style="width:100%; color:#111; font-weight:600;">Phone: <?php echo htmlspecialchars($storePhone); ?></div>
            <div style="width:100%; color:#111; font-weight:600;">Email: <?php echo htmlspecialchars($storeEmail); ?></div>
            <div style="width:100%; color:#111; font-weight:600;">Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
        </div>
        <div style="border-top:1px dashed #111; margin:8px 0; width:100%;"></div>
        <div style="width:100%; color:#111;"><b>Date:</b> <span id="waive-receipt-date"></span></div>
        <div style="width:100%; color:#111;"><b>Time:</b> <span id="waive-receipt-time"></span></div>
        <div style="width:100%; color:#111;"><b>Customer:</b> <span id="waive-receipt-customer"></span></div>
        <div style="border-top:1px dashed #111; margin:8px 0; width:100%;"></div>
        <div style="width:100%; color:#000; font-weight:700; text-align:center;">Waived Credit Items</div>
        <div id="waive-receipt-waive-details"></div>
        <div style="border-top:1px dashed #111; margin:8px 0; width:100%;"></div>
        <div style="width:100%; color:#000; font-size:11px; font-weight:700;">WAIVER SUMMARY</div>
        <div style="width:100%; color:#000;"><b>Previous Balance:</b> <span id="waive-receipt-prev-balance"></span></div>
        <div style="width:100%; color:#000;"><b>Amount Waived:</b> <span id="waive-receipt-amount"></span></div>
        <div style="width:100%; color:#000;"><b>Remaining Balance:</b> <span id="waive-receipt-balance"></span></div>
        <div style="border-top:1px solid #111; margin:4px 0; width:100%;"></div>
        <div style="width:100%; color:#000; font-size:10px;"><b>Waiver Reason:</b></div>
        <div style="width:100%; color:#111; font-size:10px; margin-bottom:4px;"><span id="waive-receipt-reason"></span></div>
        <div style="border-top:1px dashed #111; margin:8px 0; width:100%;"></div>
        <div style="margin:12px 0 4px 0; text-align:center; width:100%; color:#000; font-weight:700;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
        <div style="text-align:center; margin:8px 0; width:100%;">
            <img id="waive-receipt-qr" style="width:120px; height:120px;" />
        </div>
        <div style="border-top:1px dashed #111; margin:8px 0; width:100%;"></div>
        <div style="text-align:center; font-size:11px; width:100%; color:#111; font-weight:600;">Designed and developed by Vicotechnologies.co.ke<br>+254702214793</div>
    </div>
</template>

<!-- Edit Invoice Modal -->
<div class="modal fade" id="editInvoiceModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Invoice Items</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="editInvoiceForm" method="POST">
                    <input type="hidden" name="edit_customer_id" id="editCustomerId">
                    <div class="mb-3">
                        <label class="form-label">Customer</label>
                        <input type="text" class="form-control" id="editCustomerName" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Select Invoice to Edit</label>
                        <select class="form-select" id="editInvoiceSelect" required>
                            <option value="">Select an invoice...</option>
                        </select>
                    </div>
                    <div id="editInvoiceItems" style="display: none;">
                        <h6>Invoice Items</h6>
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Product</th>
                                        <th>Original Qty</th>
                                        <th>Original Price</th>
                                        <th>New Qty</th>
                                        <th>New Price</th>
                                        <th>New Total</th>
                                    </tr>
                                </thead>
                                <tbody id="editInvoiceItemsList">
                                    <!-- Items will be loaded dynamically -->
                                </tbody>
                            </table>
                        </div>
                        <div class="row mt-3">
                            <div class="col-md-6">
                                <strong>Original Invoice Total: <span id="originalTotal">0.00</span></strong>
                            </div>
                            <div class="col-md-6">
                                <strong>New Invoice Total: <span id="newTotal">0.00</span></strong>
                            </div>
                        </div>
                        <div class="mb-3 mt-3">
                            <label class="form-label">Reason for Edit</label>
                            <textarea class="form-control" name="edit_reason" id="editReason" rows="2" placeholder="Enter reason for editing invoice..." required></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary" id="saveInvoiceChanges" style="display: none;">Save Changes</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
// Utility function to safely get element by ID
function safeGetElement(id) {
    return document.getElementById(id) || null;
}

// Utility function to safely set element style
function safeSetStyle(element, property, value) {
    if (element && element.style) {
        element.style[property] = value;
    }
}

function showPaymentModal(creditId, amount, customerName, debtBalance) {
    document.getElementById('paymentId').value = creditId;
    document.getElementById('paymentAmount').value = amount;
    document.getElementById('customerName').value = customerName;
    
    // Store the debt balance for preview purposes if provided
    if (debtBalance !== undefined) {
        document.getElementById('paymentModal').setAttribute('data-debt-balance', parseFloat(debtBalance).toFixed(2));
    }
    
    // Fetch unpaid invoices
    fetch(`credit.php?get_unpaid_invoices=1&customer_id=${creditId}`)
        .then(response => response.json())
        .then(data => {
            const invoiceList = document.getElementById('paymentInvoiceList');
            invoiceList.innerHTML = '';
            
            if (data.invoices && data.invoices.length > 0) {
                data.invoices.forEach(invoice => {
                    const div = document.createElement('div');
                    div.className = 'form-check mb-2';
                    div.innerHTML = `
                        <input class="form-check-input" type="checkbox" name="payment_invoice_ids[]" value="${invoice.id}" id="invoice_${invoice.id}">
                        <label class="form-check-label" for="invoice_${invoice.id}">
                            Invoice #${invoice.invoice_number} - Balance: ${invoice.balance}
                        </label>
                    `;
                    invoiceList.appendChild(div);
                });
            } else {
                invoiceList.innerHTML = '<div class="text-muted">No unpaid invoices found</div>';
            }
        })
        .catch(error => {
            console.error('Error fetching unpaid invoices:', error);
            document.getElementById('paymentInvoiceList').innerHTML = '<div class="text-danger">Error loading unpaid invoices</div>';
        });
    
    const modal = new bootstrap.Modal(document.getElementById('paymentModal'));
    modal.show();
}

// Check if payment method element exists before adding event listener
var paymentMethodElement2 = document.getElementById('paymentMethod');
if (paymentMethodElement2) {
    paymentMethodElement2.addEventListener('change', function() {
        const mpesaFields = document.getElementById('mpesaFields');
        if (mpesaFields) {
            mpesaFields.style.display = this.value === 'mpesa' ? 'block' : 'none';
        }
    });
}

function printReceipt(saleId) {
    // Open receipt in a new window
    const receiptWindow = window.open(`print_receipt.php?sale_id=${saleId}`, '_blank', 'width=800,height=600');
    
    // Wait for the window to load
    receiptWindow.onload = function() {
        // Print the receipt
        receiptWindow.print();
        
        // Close the window after printing (optional)
        // receiptWindow.close();
    };
}

function printCreditReport() {
    window.print();
}

document.addEventListener('DOMContentLoaded', function() {
    document.querySelectorAll('.toggle-transactions').forEach(function(link) {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            var cid = this.getAttribute('data-customer-id');
            document.querySelectorAll('.transaction-row[data-customer-id="' + cid + '"]').forEach(function(row) {
                row.style.display = (row.style.display === 'none' || row.style.display === '') ? '' : 'none';
            });
        });
    });
    // Pay button logic
    document.querySelectorAll('.pay-btn').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var cid = this.getAttribute('data-customer-id');
            var cname = this.getAttribute('data-customer-name');
            var balance = parseFloat(this.getAttribute('data-balance')) || 0;
            
            // Set the customer ID and name
            document.getElementById('paymentCustomerId').value = cid;
            document.getElementById('customerName').value = cname;
            
            // Store the debt balance for preview purposes
            document.getElementById('paymentModal').setAttribute('data-debt-balance', balance.toFixed(2));
            
            // Set the maximum payment amount to the balance
            var amountInput = document.getElementById('paymentAmount');
            amountInput.value = balance.toFixed(2);
            amountInput.max = balance;
            
            // Clear previous invoice list
            document.getElementById('paymentInvoiceList').innerHTML = '<div class="text-muted">Loading invoices...</div>';
            
            // Fetch unpaid invoices for this customer
            fetch('credit.php?fetch_unpaid_invoices=1&customer_id=' + encodeURIComponent(cid))
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(data => {
                    var listDiv = document.getElementById('paymentInvoiceList');
                    if (data.success && Array.isArray(data.invoices) && data.invoices.length > 0) {
                        var html = '';
                        data.invoices.forEach(function(inv) {
                            html += '<div class="form-check mb-2">';
                            html += '<input class="form-check-input" type="checkbox" name="payment_invoice_ids[]" value="' + inv.id + '" data-balance="' + inv.balance_raw + '" checked>';
                            html += '<label class="form-check-label">';
                            html += 'Invoice: ' + inv.invoice_number + ' | Date: ' + (inv.sale_date ? new Date(inv.sale_date).toLocaleDateString() : 'N/A') + ' | Balance: <?php echo $currency; ?> ' + inv.balance_formatted;
                            html += '</label>';
                            html += '</div>';
                        });
                        listDiv.innerHTML = html;
                    } else {
                        listDiv.innerHTML = '<div class="text-danger">No unpaid invoices found.</div>';
                    }
                })
                .catch(error => {
                    console.error('Error fetching invoices:', error);
                    document.getElementById('paymentInvoiceList').innerHTML = 
                        '<div class="text-danger">Error loading unpaid invoices. Please try again.</div>';
                });

            // Show the payment modal
            var modal = new bootstrap.Modal(document.getElementById('paymentModal'));
            modal.show();
        });
    });

    // Update payment amount when invoices are selected/deselected
    document.getElementById('paymentInvoiceList').addEventListener('change', function(e) {
        if (e.target.type === 'checkbox') {
            var total = 0;
            var checkedBoxes = this.querySelectorAll('input[type="checkbox"]:checked');
            if (checkedBoxes.length === 0) {
                document.getElementById('paymentAmount').value = '';
                return;
            }
            checkedBoxes.forEach(function(checkbox) {
                var balance = parseFloat(checkbox.getAttribute('data-balance')) || 0;
                total += balance;
            });
            document.getElementById('paymentAmount').value = total.toFixed(2);
        }
    });

    // Handle payment method change
    var paymentMethodElement = document.getElementById('paymentMethod');
    if (paymentMethodElement) {
        paymentMethodElement.addEventListener('change', function() {
        var mpesaPhoneGroup = document.getElementById('mpesaPhoneGroup');
        var referenceGroup = document.getElementById('referenceGroup');
        
        // Show/hide M-Pesa phone number field
        if (mpesaPhoneGroup) {
            mpesaPhoneGroup.style.display = this.value === 'mpesa' ? 'block' : 'none';
        }
        
        // Show/hide reference field for Sent Via M-PESA(Received)
        if (referenceGroup) {
            referenceGroup.style.display = this.value === 'mpesa_received' ? 'block' : 'none';
        }
    });
    }
    
    // Initialize payment method fields on page load (moved outside DOMContentLoaded to avoid duplicate)
    var paymentMethod = document.getElementById('paymentMethod');
    var referenceGroup = document.getElementById('referenceGroup');
    if (paymentMethod && referenceGroup && paymentMethod.value === 'mpesa_received') {
        referenceGroup.style.display = 'block';
    }

    // Receipt button logic
    document.querySelectorAll('.receipt-btn').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var cid = this.getAttribute('data-customer-id');
            var template = document.getElementById('pending-receipt-template-' + cid);
            if (template) {
                var win = window.open('', '', 'width=400,height=600');
                win.document.write('<html><head><title>Pending Credit Receipt</title></head><body style="margin:0;">');
                win.document.write(template.innerHTML);
                win.document.write('</body></html>');
                win.document.close();
                
                // Fill in date and receipt number
                var now = new Date();
                var dateStr = now.getFullYear() + '-' + String(now.getMonth()+1).padStart(2,'0') + '-' + String(now.getDate()).padStart(2,'0') + ' ' + String(now.getHours()).padStart(2,'0') + ':' + String(now.getMinutes()).padStart(2,'0') + ':' + String(now.getSeconds()).padStart(2,'0');
                var receiptNo = 'CR-' + cid + '-' + now.getTime();
                win.document.querySelector('.receipt-date').textContent = dateStr;
                win.document.querySelector('.receipt-number').textContent = receiptNo;
                
                // Gather receipt info for QR
                var customerName = btn.getAttribute('data-customer-name') || 'Unknown Customer';
                var receiptInfo = {
                    store: '<?php echo addslashes($storeName); ?>',
                    address: '<?php echo addslashes($storeAddress); ?>',
                    phone: '<?php echo addslashes($storePhone); ?>',
                    email: '<?php echo addslashes($storeEmail); ?>',
                    website: '<?php echo addslashes($storeWebsite); ?>',
                    date: dateStr,
                    receipt_no: receiptNo,
                    customer: customerName,
                    total_pending: btn.getAttribute('data-balance') || '0.00'
                };
                
                // Add items
                receiptInfo.items = [];
                var rows = win.document.querySelectorAll('table tbody tr');
                rows.forEach(function(row) {
                    var cells = row.querySelectorAll('td');
                    if (cells.length === 4) {
                        receiptInfo.items.push({
                            date: cells[0].textContent,
                            invoice: cells[1].textContent,
                            amount: cells[2].textContent,
                            items: cells[3].textContent
                        });
                    }
                });
                
                // Generate QR code using qrserver
                var qrData = JSON.stringify(receiptInfo);
                var qrUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=' + encodeURIComponent(qrData);
                var qrImg = win.document.getElementById('receipt-qr');
                
                // Wait for QR code to load before printing
                qrImg.onload = function() {
                    setTimeout(function() {
                        win.print();
                        setTimeout(function() {
                            win.close();
                        }, 1000);
                    }, 500);
                };
                
                // Handle QR code load error
                qrImg.onerror = function() {
                    console.error('Failed to load QR code');
                    setTimeout(function() {
                        win.print();
                        setTimeout(function() {
                            win.close();
                        }, 1000);
                    }, 500);
                };
                
                // Set QR code source
                qrImg.src = qrUrl;
            }
        });
    });
    document.getElementById('showClearedCreditsBtn').addEventListener('click', function() {
        var section = document.getElementById('clearedCreditsSection');
        section.style.display = (section.style.display === 'none' || section.style.display === '') ? 'block' : 'none';
    });
    document.querySelectorAll('.toggle-cleared-transactions').forEach(function(link) {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            var cid = this.getAttribute('data-customer-id');
            document.querySelectorAll('.cleared-transaction-row[data-customer-id="' + cid + '"]').forEach(function(row) {
                row.style.display = (row.style.display === 'none' || row.style.display === '') ? '' : 'none';
            });
        });
    });
    document.querySelectorAll('.print-cleared-receipt').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var cid = this.getAttribute('data-customer-id');
            var template = document.getElementById('receipt-template-' + cid);
            if (template) {
                var win = window.open('', '', 'width=400,height=600');
                win.document.write('<html><head><title>Receipt</title></head><body style="margin:0;">');
                win.document.write(template.innerHTML);
                win.document.write('</body></html>');
                win.document.close();
                
                // Generate QR code data for cleared credits
                var qrData = JSON.stringify({
                    store: '<?php echo htmlspecialchars($storeName); ?>',
                    customer: template.getAttribute('data-customer-name'),
                    total_paid: template.getAttribute('data-total-paid'),
                    date: new Date().toLocaleDateString(),
                    time: new Date().toLocaleTimeString(),
                    status: 'Cleared'
                });
                
                // Generate QR code using QRServer
                var qrUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=' + encodeURIComponent(qrData);
                win.document.getElementById('receipt-qr').src = qrUrl;
                
                win.focus();
                setTimeout(function() { win.print(); }, 300);
            }
        });
    });
    // Toggle credit items when clicking customer name
    document.querySelectorAll('.customer-name').forEach(function(link) {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            var customerId = this.getAttribute('data-customer-id');
            var itemsRow = document.querySelector(`.credit-items[data-customer-id="${customerId}"]`);
            
            if (itemsRow) {
                if (itemsRow.style.display === 'none' || itemsRow.style.display === '') {
                    itemsRow.style.display = 'table-row';
                    this.querySelector('i').classList.remove('fa-chevron-right');
                    this.querySelector('i').classList.add('fa-chevron-down');
                } else {
                    itemsRow.style.display = 'none';
                    this.querySelector('i').classList.remove('fa-chevron-down');
                    this.querySelector('i').classList.add('fa-chevron-right');
                }
            }
        });
    });
});

// Payment message preview functionality
document.getElementById('previewPaymentMessage').addEventListener('click', function() {
    const customerName = document.getElementById('customerName').value;
    const amount = document.getElementById('paymentAmount').value;
    const debtBalance = document.getElementById('paymentModal').getAttribute('data-debt-balance');
    const paymentMethod = document.getElementById('paymentMethod').value;
    const reference = document.querySelector('input[name="reference"]').value;
    
    if (!customerName || !amount) {
        alert('Please fill in customer and amount fields to preview message');
        return;
    }
    
    // Generate preview data
    const previewData = {
        customer_name: customerName,
        amount: parseFloat(amount),
        debt_balance: parseFloat(debtBalance) || 0,
        remaining_balance: (parseFloat(debtBalance) || 0) - (parseFloat(amount) || 0),
        payment_method: paymentMethod === 'mpesa' ? 'M-Pesa' : (paymentMethod === 'mpesa_received' ? 'Sent Via M-PESA(Received)' : 'Cash'),
        reference: reference,
        store_name: '<?php echo addslashes($storeName); ?>',
        store_phone: '<?php echo addslashes($storePhone); ?>',
        store_email: '<?php echo addslashes($storeEmail); ?>',
        currency: '<?php echo $currency; ?>',
        date: new Date().toLocaleDateString(),
        time: new Date().toLocaleTimeString()
    };
    
    // Update SMS preview
    const smsMessage = `Payment Confirmation - ${previewData.store_name}

Dear ${previewData.customer_name},

Your payment has been successfully processed.

Payment Details:
- Amount Paid: ${previewData.currency} ${previewData.amount.toFixed(2)}
- Payment Method: ${previewData.payment_method}${previewData.reference ? '\n- Reference: ' + previewData.reference : ''}
- Previous Debt: ${previewData.currency} ${previewData.debt_balance.toFixed(2)}
- Remaining Balance: ${previewData.currency} ${Math.max(0, previewData.remaining_balance).toFixed(2)}
- Date: ${previewData.date}
- Time: ${previewData.time}

Thank you for your payment!

${previewData.store_name}
Phone: ${previewData.store_phone}
Email: ${previewData.store_email}`;
    
    document.getElementById('smsPreviewContent').textContent = smsMessage;
    
    // Update email preview
    const emailSubject = `Payment Confirmation - ${previewData.store_name}`;
    const emailContent = `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
            <div style="background: #f8f9fa; padding: 20px; text-align: center; border-bottom: 3px solid #007bff;">
                <h2 style="color: #007bff; margin: 0;">Payment Confirmation</h2>
                <p style="color: #6c757d; margin: 5px 0 0 0;">${previewData.store_name}</p>
            </div>
            
            <div style="padding: 30px 20px;">
                <p style="font-size: 16px; color: #333;">Dear <strong>${previewData.customer_name}</strong>,</p>
                
                <p style="color: #333; line-height: 1.6;">
                    Your payment has been successfully processed. Here are the details:
                </p>
                
                <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
                    <h3 style="color: #007bff; margin-top: 0;">Payment Details</h3>
                    <table style="width: 100%; border-collapse: collapse;">
                        <tr>
                            <td style="padding: 8px 0; color: #6c757d;">Amount Paid:</td>
                            <td style="padding: 8px 0; font-weight: bold; color: #28a745;">${previewData.currency} ${previewData.amount.toFixed(2)}</td>
                        </tr>
                        <tr>
                            <td style="padding: 8px 0; color: #6c757d;">Payment Method:</td>
                            <td style="padding: 8px 0; font-weight: bold;">${previewData.payment_method}</td>
                        </tr>${previewData.reference ? `
                        <tr>
                            <td style="padding: 8px 0; color: #6c757d;">Reference:</td>
                            <td style="padding: 8px 0; font-weight: bold;">${previewData.reference}</td>
                        </tr>` : ''}
                        <tr>
                            <td style="padding: 8px 0; color: #6c757d;">Previous Debt Balance:</td>
                            <td style="padding: 8px 0; font-weight: bold; color: #dc3545;">${previewData.currency} ${previewData.debt_balance.toFixed(2)}</td>
                        </tr>
                        <tr>
                            <td style="padding: 8px 0; color: #6c757d;">Remaining Balance:</td>
                            <td style="padding: 8px 0; font-weight: bold; color: ${Math.max(0, previewData.remaining_balance) > 0 ? '#ffc107' : '#28a745'};">${previewData.currency} ${Math.max(0, previewData.remaining_balance).toFixed(2)}</td>
                        </tr>
                        <tr>
                            <td style="padding: 8px 0; color: #6c757d;">Date & Time:</td>
                            <td style="padding: 8px 0; font-weight: bold;">${previewData.date} at ${previewData.time}</td>
                        </tr>
                    </table>
                </div>
                
                <p style="color: #333; line-height: 1.6;">
                    Thank you for your payment! If you have any questions, please don't hesitate to contact us.
                </p>
                
                <div style="text-align: center; margin: 30px 0;">
                    <div style="background: #007bff; color: white; padding: 15px; border-radius: 8px;">
                        <strong>Thank you for choosing ${previewData.store_name}!</strong>
                    </div>
                </div>
            </div>
            
            <div style="background: #f8f9fa; padding: 20px; text-align: center; color: #6c757d; font-size: 14px;">
                <p style="margin: 0;"><strong>${previewData.store_name}</strong></p>
                <p style="margin: 5px 0;">Phone: ${previewData.store_phone} | Email: ${previewData.store_email}</p>
            </div>
        </div>
    `;
    
    document.getElementById('emailSubjectPreview').textContent = emailSubject;
    document.getElementById('emailPreviewContent').innerHTML = emailContent;
    
    // Show preview modal
    const previewModal = new bootstrap.Modal(document.getElementById('paymentMessagePreviewModal'));
    previewModal.show();
});

// Add listeners for notification checkboxes to warn users about disabled services
document.addEventListener('DOMContentLoaded', function() {
    const smsCheckbox = document.getElementById('send_sms_payment');
    const emailCheckbox = document.getElementById('send_email_payment');
    const whatsappCheckbox = document.getElementById('send_whatsapp_payment');
    
    if (smsCheckbox && smsCheckbox.disabled) {
        smsCheckbox.parentElement.addEventListener('click', function(e) {
            if (e.target === smsCheckbox && smsCheckbox.disabled) {
                e.preventDefault();
                alert('SMS notifications are disabled in settings. Please enable SMS in Settings > Notifications to use this feature.');
            }
        });
    }
    
    if (emailCheckbox && emailCheckbox.disabled) {
        emailCheckbox.parentElement.addEventListener('click', function(e) {
            if (e.target === emailCheckbox && emailCheckbox.disabled) {
                e.preventDefault();
                alert('Email notifications are disabled in settings. Please enable Email in Settings > Notifications to use this feature.');
            }
        });
    }
    
    if (whatsappCheckbox && whatsappCheckbox.disabled) {
        whatsappCheckbox.parentElement.addEventListener('click', function(e) {
            if (e.target === whatsappCheckbox && whatsappCheckbox.disabled) {
                e.preventDefault();
                alert('WhatsApp notifications are disabled in settings. Please enable WhatsApp in Settings > Notifications to use this feature.');
            }
        });
    }
});

// Update the payment form submission handler
document.getElementById('paymentForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    // Get form data
    var formData = new FormData(this);
    var amount = parseFloat(formData.get('amount'));
    var customerId = formData.get('payment_customer_id');
    var customerName = document.getElementById('customerName').value;
    var paymentMethod = formData.get('payment_method');
    
    // Validate amount
    if (amount < 1) {
        document.getElementById('paymentAmount').classList.add('is-invalid');
        return;
    }

    // Show loading state
    var submitBtn = this.querySelector('button[type="submit"]');
    var originalBtnText = submitBtn.innerHTML;
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';

    // Get selected invoice IDs
    var selectedInvoices = [];
    document.querySelectorAll('#paymentInvoiceList input[type="checkbox"]:checked').forEach(function(checkbox) {
        selectedInvoices.push(checkbox.value);
    });

    if (selectedInvoices.length === 0) {
        alert('Please select at least one invoice to pay');
        submitBtn.disabled = false;
        submitBtn.innerHTML = originalBtnText;
        return;
    }

    // Add selected invoices to form data
    formData.delete('payment_invoice_ids[]'); // Remove any existing values
    selectedInvoices.forEach(function(id) {
        formData.append('payment_invoice_ids[]', id);
    });
    
    // Add notification preferences to form data
    if (document.getElementById('send_sms_payment').checked) {
        formData.append('send_sms', 'on');
    }
    if (document.getElementById('send_email_payment').checked) {
        formData.append('send_email', 'on');
    }
    if (document.getElementById('send_whatsapp_payment').checked) {
        formData.append('send_whatsapp', 'on');
    }

    // Handle M-Pesa payment
    if (paymentMethod === 'mpesa') {
        var mpesaPhone = document.getElementById('mpesaPhone').value;
        if (!mpesaPhone) {
            alert('Please enter M-Pesa phone number');
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
            return;
        }
        
        // Validate phone number
        if (!/^(?:254|\+254|0)?([7-9]{1}[0-9]{8})$/.test(mpesaPhone)) {
            alert('Please enter a valid M-Pesa phone number');
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
            return;
        }

        // Format phone number
        mpesaPhone = mpesaPhone.replace(/^(?:254|\+254|0)/, '254');

        // Call M-Pesa API
        fetch('../api/mpesa_stkpush.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ 
                phone: mpesaPhone, 
                amount: amount,
                customer_id: customerId,
                customer_name: customerName
            })
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                alert('M-Pesa payment initiated. Please check your phone to complete the payment.');
                submitPayment();
            } else {
                throw new Error(data.message || 'Failed to initiate M-Pesa payment');
            }
        })
        .catch(error => {
            console.error('M-Pesa Error:', error);
            alert('Failed to initiate M-Pesa payment: ' + error.message);
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
        });
    } else {
        // Process regular payment
        submitPayment();
    }

    function submitPayment() {
        // Log the request data for debugging
        console.log('Submitting payment with data:', {
            amount: amount,
            customerId: customerId,
            paymentMethod: paymentMethod,
            selectedInvoices: selectedInvoices
        });

        fetch(window.location.href, {
            method: 'POST',
            body: formData
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.text().then(text => {
                try {
                    const data = JSON.parse(text);
                    console.log('Server response:', data); // Debug log
                    return data;
                } catch (e) {
                    console.error('Failed to parse response:', text);
                    throw new Error('Invalid response from server');
                }
            });
        })
        .then(data => {
            if (!data || typeof data !== 'object') {
                throw new Error('Invalid response data');
            }

            if (!data.success) {
                throw new Error(data.message || 'Payment processing failed');
            }

            // Validate required data fields
            if (!data.customer || !data.amount || !data.payment_method) {
                throw new Error('Missing required payment data');
            }

            // Close the modal
            var modal = bootstrap.Modal.getInstance(document.getElementById('paymentModal'));
            modal.hide();

            // Show success message
            let successMessage = data.message;
            
            // Add notification status to success message
            if (data.notifications_sent && data.notifications_sent.length > 0) {
                successMessage += '\n\nNotifications sent: ' + data.notifications_sent.join(', ');
            }
            
            if (data.notification_errors && data.notification_errors.length > 0) {
                successMessage += '\n\nNotification errors: ' + data.notification_errors.join(', ');
            }
            
            alert(successMessage);

            // Handle WhatsApp link if generated
            if (data.whatsapp_link) {
                const openWhatsApp = confirm('WhatsApp notification link generated. Would you like to open WhatsApp to send the message?');
                if (openWhatsApp) {
                    window.open(data.whatsapp_link, '_blank');
                }
            }

            // Generate and print receipt
            var receiptWindow = window.open('', '_blank', 'width=400,height=600');
            
            // Prepare receipt data with safe defaults
            const receiptData = {
                customer: data.customer || 'N/A',
                amount: parseFloat(data.amount) || 0,
                payment_method: data.payment_method || 'N/A',
                payment_details: Array.isArray(data.payment_details) ? data.payment_details : [],
                all_invoices: Array.isArray(data.all_invoices) ? data.all_invoices : [],
                remaining_balance: parseFloat(data.remaining_balance) || 0,
                previous_balance: parseFloat(data.previous_balance) || 0,
                total_original_credit: parseFloat(data.total_original_credit) || 0,
                total_paid: parseFloat(data.total_paid) || 0,
                waived_amount: parseFloat(data.waived_amount) || 0
            };

            // Write receipt content
            receiptWindow.document.write(`
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Payment Receipt</title>
                    <meta charset="UTF-8">
                    <style>
                        @page {
                            size: 80mm auto;
                            margin: 0;
                        }
                        body { 
                            font-family: 'Arial Black', Arial, sans-serif;
                            font-size: 12px;
                            color: #000;
                            margin: 0;
                            padding: 5px;
                            width: 80mm;
                            max-width: 80mm;
                        }
                        .header { 
                            text-align: center;
                            margin-bottom: 5px;
                            font-size: 11px;
                        }
                        .header .store-name {
                            font-size: 13px;
                            font-weight: bold;
                            margin-bottom: 2px;
                        }
                        .divider { 
                            border-top: 1px dashed #000;
                            margin: 5px 0;
                        }
                        .payment-details { 
                            margin: 5px 0;
                            font-size: 11px;
                        }
                        table {
                            width: 100%;
                            border-collapse: collapse;
                        }
                        th, td {
                            padding: 2px;
                            text-align: left;
                        }
                        th {
                            border-bottom: 1px solid #000;
                        }
                        .thank-you { 
                            text-align: center;
                            margin: 8px 0;
                            font-weight: bold;
                        }
                        .qr-code { 
                            text-align: center;
                            margin: 8px 0;
                        }
                        .qr-code img {
                            width: 100px;
                            height: 100px;
                        }
                    </style>
                </head>
                <body>
                    <div style="text-align:center; font-weight:900; margin-bottom:2px; font-size:1.1rem;">PAYMENT RECEIPT</div>
                    <?php if ($logoData): ?>
                        <?php
                        $mimeType = "image/jpeg"; // Default to JPEG
                        if (strpos(strtolower($logoName), ".png") !== false) {
                            $mimeType = "image/png";
                        } elseif (strpos(strtolower($logoName), ".gif") !== false) {
                            $mimeType = "image/gif";
                        }
                        ?>
                        <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
                    <?php endif; ?>
                    <div class="header">
                        <div class="store-name"><?php echo htmlspecialchars($storeName); ?></div>
                        <div><?php echo htmlspecialchars($storeAddress); ?></div>
                        <div>Tel: <?php echo htmlspecialchars($storePhone); ?></div>
                        <div>Email: <?php echo htmlspecialchars($storeEmail); ?></div>
                        <div>Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div>Date: ${new Date().toLocaleDateString()}</div>
                        <div>Time: ${new Date().toLocaleTimeString()}</div>
                        <div>Customer: ${receiptData.customer}</div>
                        <div>Payment Method: ${receiptData.payment_method}</div>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div style="font-weight: bold; margin-bottom: 8px; font-family: Arial Black, Arial, sans-serif;">Credit Items Payment Details:</div>
                        <table style="width: 100%; font-size: 10px; border-collapse: collapse; margin-bottom: 10px; font-family: Arial Black, Arial, sans-serif;">
                            <thead>
                                <tr style="background: #f1f5f9; font-family: Arial Black, Arial, sans-serif; font-weight: 700;">
                                    <th style="text-align: left; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Items</th>
                                    <th style="text-align: center; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Qty</th>
                                    <th style="text-align: left; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Payment Details</th>
                                    <th style="text-align: right; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Waived</th>
                                    <th style="text-align: right; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Credit Amount</th>
                                    <th style="text-align: right; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Paid</th>
                                    <th style="text-align: right; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Balance</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${receiptData.all_invoices.map((invoice, invoiceIndex) => {
                                    // Show all items for each invoice
                                    if (invoice.items && invoice.items.length > 0) {
                                        return invoice.items.map((item, itemIndex) => `
                                            <tr style="background: ${invoice.is_being_paid ? '#e6f7ff' : (invoiceIndex % 2 === 0 ? '#fff' : '#f8fafc')}; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; ${invoice.is_being_paid ? 'border-left: 3px solid #1890ff;' : ''}">
                                                <td style="padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">
                                                    ${item.product_name}
                                                    ${item.serial_numbers ? '<br><small style="font-size: 6px;">SN: ' + item.serial_numbers + '</small>' : ''}
                                                    ${itemIndex === 0 ? '<br><small style="font-size: 6px; color: #666;">INV-' + invoice.invoice_number.replace(/[^\d]/g, '') + (invoice.is_being_paid ? ' <span style="color: #1890ff; font-weight: bold;">(PAYING NOW)</span>' : '') + '</small>' : ''}
                                                </td>
                                                <td style="text-align: center; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${item.quantity}</td>
                                                <td style="padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${itemIndex === 0 ? (invoice.payment_method_details || 'Credit Sale') : ''}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #dc2626; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${itemIndex === 0 && invoice.previously_waived > 0 ? (parseFloat(invoice.previously_waived) || 0).toFixed(2) : (itemIndex === 0 ? '-' : '')}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${itemIndex === 0 ? (parseFloat(invoice.credit_amount) || 0).toFixed(2) : ''}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${itemIndex === 0 ? (parseFloat(invoice.previously_paid) + parseFloat(invoice.current_payment)).toFixed(2) : ''}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${itemIndex === 0 ? (parseFloat(invoice.balance_after) || 0).toFixed(2) : ''}</td>
                                            </tr>
                                        `).join('');
                                    } else {
                                        // Fallback if no detailed items
                                        return `
                                            <tr style="background: ${invoice.is_being_paid ? '#e6f7ff' : (invoiceIndex % 2 === 0 ? '#fff' : '#f8fafc')}; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; ${invoice.is_being_paid ? 'border-left: 3px solid #1890ff;' : ''}">
                                                <td style="padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">
                                                    No items found
                                                    <br><small style="font-size: 6px; color: #666;">INV-${invoice.invoice_number.replace(/[^\d]/g, '')}${invoice.is_being_paid ? ' <span style="color: #1890ff; font-weight: bold;">(PAYING NOW)</span>' : ''}</small>
                                                </td>
                                                <td style="text-align: center; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${invoice.total_quantity || 0}</td>
                                                <td style="padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${invoice.payment_method_details || 'Credit Sale'}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #dc2626; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${invoice.previously_waived > 0 ? (parseFloat(invoice.previously_waived) || 0).toFixed(2) : '-'}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${(parseFloat(invoice.credit_amount) || 0).toFixed(2)}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${(parseFloat(invoice.previously_paid) + parseFloat(invoice.current_payment)).toFixed(2)}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${(parseFloat(invoice.balance_after) || 0).toFixed(2)}</td>
                                            </tr>
                                        `;
                                    }
                                }).join('')}
                            </tbody>
                        </table>
                        <div style="text-align: center; font-size: 10px; margin-top: 5px; padding-top: 5px; border-top: 1px solid #000; font-family: Arial Black, Arial, sans-serif; font-weight: 700;">
                            <strong>Total Payment Applied: ${receiptData.amount.toFixed(2)}</strong>
                        </div>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div style="display: flex; justify-content: space-between;">
                            <span>Total Credit (Original):</span>
                            <span>${(receiptData.total_original_credit || receiptData.previous_balance).toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span>Amount Paid (-):</span>
                            <span>-${receiptData.amount.toFixed(2)}</span>
                        </div>
                        ${receiptData.waived_amount > 0 ? `
                        <div style="display: flex; justify-content: space-between; color: #ff6b6b;">
                            <span>Amount Waived (-):</span>
                            <span>-${receiptData.waived_amount.toFixed(2)}</span>
                        </div>` : ''}
                        <div style="display: flex; justify-content: space-between;">
                            <span>Total Paid:</span>
                            <span>${receiptData.total_paid.toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between; font-weight: bold; border-top: 1px dashed #000; padding-top: 3px; margin-top: 3px;">
                            <span>Remaining Balance:</span>
                            <span>${(receiptData.previous_balance - receiptData.amount - receiptData.waived_amount).toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span>Payment Method:</span>
                            <span>${receiptData.payment_method}</span>
                        </div>
                    </div>
                    <div class="divider"></div>
                    <div class="thank-you">
                        Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!
                    </div>
                    <div class="qr-code">
                        <img id="receipt-qr" src="" alt="Payment QR Code">
                    </div>
                    <div class="divider"></div>
                    <div style="text-align: center; font-size: 10px;">
                        Designed and developed by Vicotechnologies.co.ke<br>+254702214793
                    </div>
                </body>
                </html>
            `);
            receiptWindow.document.close();

            // Generate QR code data (simplified for better scanning)
            const qrData = JSON.stringify({
                type: 'PAYMENT',
                store: '<?php echo addslashes($storeName); ?>',
                store_phone: '<?php echo addslashes($storePhone); ?>',
                store_email: '<?php echo addslashes($storeEmail); ?>',
                customer: receiptData.customer,
                payment_amount: receiptData.amount,
                total_paid: receiptData.total_paid || '0',
                remaining_balance: receiptData.remaining_balance,
                date: new Date().toISOString().split('T')[0],
                time: new Date().toLocaleTimeString('en-GB', {hour12: false}),
                receipt: 'PAY-' + new Date().getTime()
            });

            // Set QR code image source
            const qrImg = receiptWindow.document.getElementById('receipt-qr');
            qrImg.src = `https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=${encodeURIComponent(qrData)}`;

            // Wait for QR code to load before printing
            qrImg.onload = function() {
                setTimeout(() => {
                    receiptWindow.print();
                    setTimeout(() => window.location.reload(), 1000);
                }, 500);
            };

            // Fallback in case QR code fails to load
            qrImg.onerror = function() {
                console.error('Failed to load QR code');
                setTimeout(() => {
                    receiptWindow.print();
                    setTimeout(() => window.location.reload(), 1000);
                }, 500);
            };
        })
        .catch(error => {
            console.error('Payment Error:', error);
            alert('An error occurred while processing your request: ' + error.message);
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
        });
    }
});

// ... existing code ...
    // Waive button logic
    document.querySelectorAll('.waive-btn').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var cid = this.getAttribute('data-customer-id');
            var cname = this.getAttribute('data-customer-name');
            var balance = parseFloat(this.getAttribute('data-balance')) || 0;
            
            // Set the customer ID and name
            document.getElementById('waiveCustomerId').value = cid;
            document.getElementById('waiveCustomerName').value = cname;
            
            // Set the maximum waive amount to the balance
            var amountInput = document.getElementById('waiveAmount');
            amountInput.value = balance.toFixed(2);
            amountInput.max = balance;
            
            // Clear previous invoice list
            document.getElementById('waiveInvoiceList').innerHTML = '<div class="text-muted">Loading invoices...</div>';
            
            // Fetch unpaid invoices for this customer
            fetch('credit.php?fetch_unpaid_invoices=1&customer_id=' + encodeURIComponent(cid))
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(data => {
                    console.log('Received unpaid invoices data:', data);
                    var listDiv = document.getElementById('waiveInvoiceList');
                    if (data.success && Array.isArray(data.invoices) && data.invoices.length > 0) {
                        var html = '';
                        data.invoices.forEach(function(inv) {
                            console.log('Processing invoice:', inv);
                            console.log('Balance raw:', inv.balance_raw, 'Type:', typeof inv.balance_raw);
                            console.log('Balance formatted:', inv.balance_formatted);
                            html += '<div class="form-check mb-2">';
                            html += '<input class="form-check-input waive-invoice-checkbox" type="checkbox" name="waive_invoice_ids[]" value="' + inv.id + '" data-balance="' + inv.balance_raw + '" checked>';
                            html += '<label class="form-check-label">';
                            html += 'Invoice: ' + inv.invoice_number + ' | Date: ' + (inv.sale_date ? new Date(inv.sale_date).toLocaleDateString() : 'N/A') + ' | Balance: <?php echo $currency; ?> ' + inv.balance_formatted;
                            html += '</label>';
                            html += '</div>';
                        });
                        listDiv.innerHTML = html;

                        // Add event listeners to checkboxes to update waive amount
                        updateWaiveAmountFromCheckboxes();
                        document.querySelectorAll('.waive-invoice-checkbox').forEach(function(checkbox) {
                            checkbox.addEventListener('change', updateWaiveAmountFromCheckboxes);
                        });
                    } else {
                        listDiv.innerHTML = '<div class="text-danger">No unpaid invoices found.</div>';
                    }
                })
                .catch(error => {
                    console.error('Error fetching invoices:', error);
                    document.getElementById('waiveInvoiceList').innerHTML = 
                        '<div class="text-danger">Error loading unpaid invoices. Please try again.</div>';
                });

            // Show the waive modal
            var modal = new bootstrap.Modal(document.getElementById('waiveModal'));
            modal.show();
        });
    });

    // Handle waive form submission
    document.getElementById('waiveForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Get form data
        var formData = new FormData(this);
        var amount = parseFloat(formData.get('waive_amount'));
        var customerId = formData.get('waive_customer_id');
        var reason = formData.get('waive_reason');
        
        // Validate amount
        if (amount <= 0) {
            document.getElementById('waiveAmount').classList.add('is-invalid');
            return;
        }

        // Show loading state
        var submitBtn = this.querySelector('button[type="submit"]');
        var originalBtnText = submitBtn.innerHTML;
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';

        // Get selected invoice IDs
        var selectedInvoices = [];
        document.querySelectorAll('#waiveInvoiceList input[type="checkbox"]:checked').forEach(function(checkbox) {
            selectedInvoices.push(checkbox.value);
        });

        if (selectedInvoices.length === 0) {
            alert('Please select at least one invoice to waive');
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
            return;
        }

        // Add selected invoices to form data
        formData.delete('waive_invoice_ids[]'); // Remove any existing values
        selectedInvoices.forEach(function(id) {
            formData.append('waive_invoice_ids[]', id);
        });

        // Log the form data for debugging
        console.log('Submitting waive request with data:', {
            amount: amount,
            customerId: customerId,
            reason: reason,
            selectedInvoices: selectedInvoices
        });

        // Submit the form
        fetch(window.location.href, {
            method: 'POST',
            body: formData
        })
        .then(response => {
            console.log('Server response status:', response.status);
            return response.text().then(text => {
                console.log('Raw server response:', text);
                try {
                    // Try to parse the response as JSON
                    const data = JSON.parse(text);
                    if (!data || typeof data !== 'object') {
                        throw new Error('Invalid response format');
                    }
                    return data;
                } catch (e) {
                    console.error('Failed to parse response:', e);
                    // If the response is not JSON, check if it contains an error message
                    if (text.includes('error') || text.includes('Error')) {
                        throw new Error(text);
                    }
                    throw new Error('Server returned an invalid response. Please try again.');
                }
            });
        })
        .then(data => {
            if (!data.success) {
                throw new Error(data.message || 'Waive processing failed');
            }

            // Close the modal
            var modal = bootstrap.Modal.getInstance(document.getElementById('waiveModal'));
            modal.hide();

            // Show success message
            alert(data.message || 'Amount waived successfully');

            // Generate and print receipt
            var receiptWindow = window.open('', '_blank', 'width=400,height=600');
            
            // Prepare receipt data
            const receiptData = {
                customer: document.getElementById('waiveCustomerName').value,
                amount: amount,
                reason: reason,
                previous_balance: data.old_balance,
                total_original_credit: data.total_original_credit,
                old_balance: data.old_balance,
                waived_amount: data.waived_amount,
                remaining_balance: data.remaining_balance,
                waive_details: data.waive_details || [],
                all_unpaid_invoices: data.all_unpaid_invoices || []
            };

            // Write receipt content
            receiptWindow.document.write(`
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Waiver Receipt</title>
                    <meta charset="UTF-8">
                    <style>
                        @page {
                            size: 80mm auto;
                            margin: 0;
                        }
                        body { 
                            font-family: 'Arial Black', Arial, sans-serif;
                            font-size: 12px;
                            color: #000;
                            margin: 0;
                            padding: 5px;
                            width: 80mm;
                            max-width: 80mm;
                            font-weight: 900;
                        }
                        .header { 
                            text-align: center;
                            margin-bottom: 5px;
                            font-size: 11px;
                        }
                        .header .store-name {
                            font-size: 13px;
                            font-weight: bold;
                            margin-bottom: 2px;
                        }
                        .divider { 
                            border-top: 1px dashed #000;
                            margin: 5px 0;
                        }
                        .payment-details { 
                            margin: 5px 0;
                            font-size: 11px;
                        }
                        table {
                            width: 100%;
                            border-collapse: collapse;
                        }
                        th, td {
                            padding: 2px;
                            text-align: left;
                        }
                        th {
                            border-bottom: 1px solid #000;
                        }
                        .thank-you { 
                            text-align: center;
                            margin: 8px 0;
                            font-weight: bold;
                        }
                        .qr-code { 
                            text-align: center;
                            margin: 8px 0;
                        }
                        .qr-code img {
                            width: 100px;
                            height: 100px;
                        }
                    </style>
                </head>
                <body>
                    <div style="text-align:center; font-weight:900; margin-bottom:2px; font-size:1.1rem;">WAIVER RECEIPT</div>
                    <?php if ($logoData): ?>
                        <?php
                        $mimeType = 'image/jpeg'; // Default to JPEG
                        if (strpos(strtolower($logoName), '.png') !== false) {
                            $mimeType = 'image/png';
                        } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                            $mimeType = 'image/gif';
                        }
                        ?>
                        <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
                    <?php endif; ?>
                    <div class="header">
                        <div class="store-name"><?php echo htmlspecialchars($storeName); ?></div>
                        <div><?php echo htmlspecialchars($storeAddress); ?></div>
                        <div>Tel: <?php echo htmlspecialchars($storePhone); ?></div>
                        <div>Email: <?php echo htmlspecialchars($storeEmail); ?></div>
                        <div>Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div>Date: ${new Date().toLocaleDateString()}</div>
                        <div>Time: ${new Date().toLocaleTimeString()}</div>
                        <div>Customer: ${receiptData.customer}</div>
                    </div>
                    <div class="divider"></div>
                    <div class="payment-details">
                        <div style="font-weight: bold; margin-bottom: 8px; font-family: Arial Black, Arial, sans-serif;">Credit Items Waiver Details:</div>
                        <table style="width: 100%; font-size: 10px; border-collapse: collapse; margin-bottom: 10px; font-family: Arial Black, Arial, sans-serif;">
                            <thead>
                                <tr style="background: #f1f5f9; font-family: Arial Black, Arial, sans-serif; font-weight: 700;">
                                    <th style="text-align: left; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Items</th>
                                    <th style="text-align: center; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Qty</th>
                                    <th style="text-align: left; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Payment Details</th>
                                    <th style="text-align: right; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Waived</th>
                                    <th style="text-align: right; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Credit Amount</th>
                                    <th style="text-align: right; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Paid</th>
                                    <th style="text-align: right; padding: 4px 2px; font-size: 9px; font-weight: 700; border-bottom: 1px solid #000; color: #111; font-family: Arial Black, Arial, sans-serif;">Balance</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${receiptData.waive_details.map((detail, detailIndex) => {
                                    // If all_items is available, show individual items
                                    if (detail.all_items && detail.all_items.length > 0) {
                                        return detail.all_items.map((item, itemIndex) => `
                                            <tr style="background: ${detailIndex % 2 === 0 ? '#fff' : '#f8fafc'}; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700;">
                                                <td style="padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">
                                                    ${item.product_name}
                                                    ${item.serial_numbers ? '<br><small style="font-size: 6px;">SN: ' + item.serial_numbers + '</small>' : ''}
                                                    ${itemIndex === 0 ? '<br><small style="font-size: 6px; color: #666;">Invoice: ' + detail.invoice_number + '</small>' : ''}
                                                </td>
                                                <td style="text-align: center; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${item.quantity}</td>
                                                <td style="padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${itemIndex === 0 ? (detail.payment_method_details || 'Credit Sale') : ''}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #dc2626; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${itemIndex === 0 ? (parseFloat(detail.previously_waived) + parseFloat(detail.waived_amount)).toFixed(2) : ''}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${itemIndex === 0 ? (parseFloat(detail.invoice_total) || 0).toFixed(2) : ''}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${itemIndex === 0 ? (parseFloat(detail.previously_paid) || 0).toFixed(2) : ''}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${itemIndex === 0 ? (parseFloat(detail.balance_after) || 0).toFixed(2) : ''}</td>
                                            </tr>
                                        `).join('');
                                    } else {
                                        // Fallback to original format if no detailed items
                                        return `
                                            <tr style="background: ${detailIndex % 2 === 0 ? '#fff' : '#f8fafc'}; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700;">
                                                <td style="padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${detail.items || 'No items found'}</td>
                                                <td style="text-align: center; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${detail.total_quantity || 0}</td>
                                                <td style="padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${detail.payment_method_details || 'Credit Sale'}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #dc2626; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${(parseFloat(detail.previously_waived) + parseFloat(detail.waived_amount)).toFixed(2)}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${(parseFloat(detail.invoice_total) || 0).toFixed(2)}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${(parseFloat(detail.previously_paid) || 0).toFixed(2)}</td>
                                                <td style="text-align: right; padding: 3px 2px; color: #111; font-family: Arial Black, Arial, sans-serif; font-weight: 700; font-size: 8px;">${(parseFloat(detail.balance_after) || 0).toFixed(2)}</td>
                                            </tr>
                                        `;
                                    }
                                }).join('')}
                            </tbody>
                        </table>
                        <div style="text-align: center; font-size: 10px; margin-top: 5px; padding-top: 5px; border-top: 1px solid #000; font-family: Arial Black, Arial, sans-serif; font-weight: 700;">
                            <strong>Total Amount Waived: ${receiptData.waived_amount.toFixed(2)}</strong>
                        </div>
                    </div>
                    <div class="divider"></div>
                    ${receiptData.all_unpaid_invoices && receiptData.all_unpaid_invoices.length > 0 ? `
                        <div style="font-weight: bold; text-align: center; margin-bottom: 4px; font-size: 12px;">UNPAID CREDIT ITEMS</div>
                        ${receiptData.all_unpaid_invoices.map(invoice => `
                            <div style="border: 1px solid #ddd; margin-bottom: 6px; padding: 3px; font-size: 9px;">
                                <div style="font-weight: bold; margin-bottom: 2px; text-align: center; font-size: 10px;">
                                    Invoice: ${invoice.invoice_number} - ${new Date(invoice.sale_date).toLocaleDateString()}
                                    ${invoice.status ? ' (' + invoice.status + ')' : ''}
                                </div>
                                
                                <!-- Items Table -->
                                <table style="width: 100%; font-size: 8px; border-collapse: collapse; margin-bottom: 3px;">
                                    <thead>
                                        <tr style="border-bottom: 1px solid #999;">
                                            <th style="text-align: left; padding: 1px;">Item</th>
                                            <th style="text-align: center; padding: 1px;">Qty</th>
                                            <th style="text-align: right; padding: 1px;">Price</th>
                                            <th style="text-align: right; padding: 1px;">Total</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        ${invoice.items.map(item => `
                                            <tr>
                                                <td style="text-align: left; padding: 1px; font-size: 7px;">
                                                    ${item.product_name}
                                                    ${item.serial_numbers ? '<br><small style="font-size: 6px;">SN: ' + item.serial_numbers + '</small>' : ''}
                                                </td>
                                                <td style="text-align: center; padding: 1px;">${item.quantity}</td>
                                                <td style="text-align: right; padding: 1px;">${parseFloat(item.price).toFixed(2)}</td>
                                                <td style="text-align: right; padding: 1px;">${(parseFloat(item.price) * parseInt(item.quantity)).toFixed(2)}</td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                                
                                <!-- Financial Summary for this invoice -->
                                <div style="font-size: 8px; text-align: right; border-top: 1px dashed #999; padding-top: 2px;">
                                    <div style="font-weight: bold;">Total Amount: ${parseFloat(invoice.credit_amount).toFixed(2)}</div>
                                    ${invoice.pos_paid_amount > 0 ? `<div>Paid (${invoice.payment_methods.join(', ')}): ${parseFloat(invoice.pos_paid_amount).toFixed(2)}</div>` : ''}
                                    ${invoice.paid > 0 ? `<div>[Credit] Paid: ${parseFloat(invoice.paid).toFixed(2)}</div>` : ''}
                                    ${invoice.waived > 0 ? `<div>Waived: ${parseFloat(invoice.waived).toFixed(2)}</div>` : ''}
                                    <div style="font-weight: bold; color: #e11d48;">Remaining Balance: ${parseFloat(invoice.balance).toFixed(2)}</div>
                    </div>
                            </div>
                        `).join('')}
                    <div class="divider"></div>
                    ` : ''}
                    <div class="payment-details">
                        ${receiptData.waive_details.some(detail => detail.pos_paid_amount > 0) ? `
                            <div style="font-weight: bold; margin-bottom: 3px; font-size: 11px;">POS PAYMENTS MADE:</div>
                            ${receiptData.waive_details.filter(detail => detail.pos_paid_amount > 0).map(detail => `
                                <div style="display: flex; justify-content: space-between; font-size: 10px; margin-bottom: 2px;">
                                    <span>Invoice ${detail.invoice_number} (${detail.payment_methods.join(', ')}):</span>
                                    <span>${detail.pos_paid_amount.toFixed(2)}</span>
                                </div>
                            `).join('')}
                            <div class="divider"></div>
                        ` : ''}
                        <div style="display: flex; justify-content: space-between;">
                            <span>Total Credit (Original):</span>
                            <span>${(receiptData.total_original_credit || receiptData.old_balance).toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span>Previous Balance:</span>
                            <span>${(receiptData.old_balance || receiptData.previous_balance).toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span>Amount Waived:</span>
                            <span>${receiptData.waived_amount.toFixed(2)}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between;">
                            <span>Remaining Balance:</span>
                            <span>${receiptData.remaining_balance.toFixed(2)}</span>
                        </div>
                        <div style="margin-top: 5px;">
                            <div><b>Reason for Waiver:</b></div>
                            <div>${receiptData.reason}</div>
                        </div>
                    </div>
                    <div class="divider"></div>
                    <div class="thank-you">
                        Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!
                    </div>
                    <div class="qr-code">
                        <img id="receipt-qr" src="" alt="Waiver QR Code">
                    </div>
                    <div class="divider"></div>
                    <div style="text-align: center; font-size: 10px;">
                        Designed and developed by Vicotechnologies.co.ke<br>+254702214793
                    </div>
                </body>
                </html>
            `);
            receiptWindow.document.close();

            // Generate QR code data (simplified for better scanning)
            const qrData = JSON.stringify({
                type: 'WAIVER',
                store: '<?php echo addslashes($storeName); ?>',
                store_phone: '<?php echo addslashes($storePhone); ?>',
                store_email: '<?php echo addslashes($storeEmail); ?>',
                customer: receiptData.customer,
                waived_amount: receiptData.waived_amount,
                total_paid: receiptData.total_paid || '0',
                remaining_balance: receiptData.remaining_balance,
                reason: receiptData.reason,
                date: new Date().toISOString().split('T')[0],
                time: new Date().toLocaleTimeString('en-GB', {hour12: false}),
                receipt: 'WAIVE-' + new Date().getTime()
            });

            // Set QR code image source
            const qrImg = receiptWindow.document.getElementById('receipt-qr');
            qrImg.src = `https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=${encodeURIComponent(qrData)}`;

            // Wait for QR code to load before printing
            qrImg.onload = function() {
                setTimeout(() => {
                    receiptWindow.print();
                    setTimeout(() => window.location.reload(), 1000);
                }, 500);
            };

            // Fallback in case QR code fails to load
            qrImg.onerror = function() {
                console.error('Failed to load QR code');
                setTimeout(() => {
                    receiptWindow.print();
                    setTimeout(() => window.location.reload(), 1000);
                }, 500);
            };
        })
        .catch(error => {
            console.error('Waive Error:', error);
            alert('An error occurred while processing your request: ' + error.message);
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
        });
    });
// ... existing code ...

    // Edit Invoice button logic
    document.querySelectorAll('.edit-invoice-btn').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var cid = this.getAttribute('data-customer-id');
            var cname = this.getAttribute('data-customer-name');
            
            // Set the customer ID and name
            document.getElementById('editCustomerId').value = cid;
            document.getElementById('editCustomerName').value = cname;
            
            // Clear previous invoice options
            var selectElement = document.getElementById('editInvoiceSelect');
            selectElement.innerHTML = '<option value="">Select an invoice...</option>';
            
            // Hide items section initially
            var editInvoiceItems = document.getElementById('editInvoiceItems');
            var saveInvoiceChanges = document.getElementById('saveInvoiceChanges');
            if (editInvoiceItems) editInvoiceItems.style.display = 'none';
            if (saveInvoiceChanges) saveInvoiceChanges.style.display = 'none';
            
            // Fetch invoices for this customer
            fetch('credit.php?fetch_invoice_for_edit=1&customer_id=' + encodeURIComponent(cid))
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.invoices && data.invoices.length > 0) {
                        data.invoices.forEach(function(invoice) {
                            var option = document.createElement('option');
                            option.value = invoice.id;
                            option.textContent = 'Invoice #' + invoice.invoice_number + ' - Balance: ' + invoice.balance;
                            selectElement.appendChild(option);
                        });
                    } else {
                        selectElement.innerHTML = '<option value="">No unpaid invoices found</option>';
                    }
                })
                .catch(error => {
                    console.error('Error fetching invoices:', error);
                    selectElement.innerHTML = '<option value="">Error loading invoices</option>';
                });
            
            // Show the edit modal
            var modal = new bootstrap.Modal(document.getElementById('editInvoiceModal'));
            modal.show();
        });
    });

    // Handle invoice selection change
    document.getElementById('editInvoiceSelect').addEventListener('change', function() {
        var invoiceId = this.value;
        var itemsDiv = document.getElementById('editInvoiceItems');
        var saveBtn = document.getElementById('saveInvoiceChanges');
        
        if (!invoiceId) {
            if (itemsDiv) itemsDiv.style.display = 'none';
            if (saveBtn) saveBtn.style.display = 'none';
            return;
        }
        
        // Fetch invoice items
        fetch('credit.php?fetch_invoice_items=1&invoice_id=' + invoiceId)
            .then(response => response.json())
            .then(data => {
                if (data.success && data.items) {
                    var tbody = document.getElementById('editInvoiceItemsList');
                    tbody.innerHTML = '';
                    
                    var originalTotal = parseFloat(data.total_amount) || 0;
                    document.getElementById('originalTotal').textContent = originalTotal.toFixed(2);
                    
                    data.items.forEach(function(item) {
                        var row = document.createElement('tr');
                        row.innerHTML = `
                            <td>${item.product_name}</td>
                            <td>${item.quantity}</td>
                            <td>${parseFloat(item.price).toFixed(2)}</td>
                            <td>
                                <input type="number" class="form-control form-control-sm qty-input" 
                                       name="items[${item.id}][quantity]" 
                                       value="${item.quantity}" min="0" step="1" 
                                       data-item-id="${item.id}">
                            </td>
                            <td>
                                <input type="number" class="form-control form-control-sm price-input" 
                                       name="items[${item.id}][price]" 
                                       value="${parseFloat(item.price).toFixed(2)}" min="0" step="0.01" 
                                       data-item-id="${item.id}">
                            </td>
                            <td class="item-total">${parseFloat(item.total).toFixed(2)}</td>
                        `;
                        tbody.appendChild(row);
                    });
                    
                    // Add event listeners to recalculate totals
                    tbody.querySelectorAll('.qty-input, .price-input').forEach(function(input) {
                        input.addEventListener('input', calculateNewTotal);
                    });
                    
                    // Initial calculation
                    calculateNewTotal();
                    
                    if (itemsDiv) itemsDiv.style.display = 'block';
                    if (saveBtn) saveBtn.style.display = 'inline-block';
                } else {
                    alert('Error loading invoice items: ' + (data.message || 'Unknown error'));
                }
            })
            .catch(error => {
                console.error('Error fetching invoice items:', error);
                alert('Error loading invoice items');
            });
    });

    // Function to calculate new total
    function calculateNewTotal() {
        var newTotal = 0;
        document.querySelectorAll('#editInvoiceItemsList tr').forEach(function(row) {
            var qtyInput = row.querySelector('.qty-input');
            var priceInput = row.querySelector('.price-input');
            var totalCell = row.querySelector('.item-total');
            
            if (qtyInput && priceInput && totalCell) {
                var qty = parseFloat(qtyInput.value) || 0;
                var price = parseFloat(priceInput.value) || 0;
                var itemTotal = qty * price;
                
                totalCell.textContent = itemTotal.toFixed(2);
                newTotal += itemTotal;
            }
        });
        
        document.getElementById('newTotal').textContent = newTotal.toFixed(2);
    }

    // Handle edit invoice form submission
    document.getElementById('editInvoiceForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        var formData = new FormData(this);
        var invoiceId = document.getElementById('editInvoiceSelect').value;
        var reason = document.getElementById('editReason').value;
        
        if (!invoiceId) {
            alert('Please select an invoice to edit');
            return;
        }
        
        if (!reason.trim()) {
            alert('Please provide a reason for editing the invoice');
            return;
        }
        
        // Add invoice ID to form data
        formData.append('edit_invoice_id', invoiceId);
        
        // Show loading state
        var submitBtn = document.getElementById('saveInvoiceChanges');
        var originalBtnText = submitBtn.innerHTML;
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving...';
        
        // Submit the form
        fetch(window.location.href, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Close the modal
                var modal = bootstrap.Modal.getInstance(document.getElementById('editInvoiceModal'));
                modal.hide();
                
                // Show success message
                alert('Invoice updated successfully! Original Total: ' + data.original_total + ', New Total: ' + data.new_total);
                
                // Reload the page to show updated data
                window.location.reload();
            } else {
                alert('Error: ' + (data.message || 'Failed to update invoice'));
            }
        })
        .catch(error => {
            console.error('Error updating invoice:', error);
            alert('Error updating invoice: ' + error.message);
        })
        .finally(() => {
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
        });
    });

// Add this function to update waive amount
function updateWaiveAmountFromCheckboxes() {
    var total = 0;
    var debug = [];
    document.querySelectorAll('.waive-invoice-checkbox:checked').forEach(function(checkbox) {
        var balance = checkbox.getAttribute('data-balance');
        var parsedBalance = parseFloat(balance) || 0;
        debug.push('Balance: "' + balance + '" -> Parsed: ' + parsedBalance);
        total += parsedBalance;
    });
    
    // Debug logging
    console.log('Waive amount calculation debug:');
    console.log('Found checkboxes:', document.querySelectorAll('.waive-invoice-checkbox:checked').length);
    debug.forEach(function(msg) {
        console.log(msg);
    });
    console.log('Total calculated:', total);
    
    var amountInput = document.getElementById('waiveAmount');
    if (amountInput) {
        amountInput.value = total.toFixed(2);
        amountInput.max = total;
    }
}
</script>

<!-- Show waive debug info if set -->
<?php if (isset($_SESSION['waive_debug'])): ?>
<div style="background:#fff3cd; color:#856404; border:1px solid #ffeeba; padding:12px; margin:16px 0; border-radius:6px; font-size:1.1rem;">
    <b>Waive Debug Info:</b><br>
    Old Balance: <?php echo number_format($_SESSION['waive_debug']['old_balance'], 2); ?><br>
    Waived Amount: <?php echo number_format($_SESSION['waive_debug']['waived_amount'], 2); ?><br>
    New Balance: <?php echo number_format($_SESSION['waive_debug']['new_balance'], 2); ?><br>
    <?php if (!empty($_SESSION['waive_debug']['waive_debug_details'])): ?>
    <b>Details:</b><ul>
        <?php foreach ($_SESSION['waive_debug']['waive_debug_details'] as $detail): ?>
        <li><?php echo htmlspecialchars($detail); ?></li>
        <?php endforeach; ?>
    </ul>
    <?php endif; ?>
</div>
<?php unset($_SESSION['waive_debug']); endif; ?>

<!-- Show pay debug info if set -->
<?php if (isset($_SESSION['pay_debug'])): ?>
<div style="background:#e0f7fa; color:#006064; border:1px solid #b2ebf2; padding:12px; margin:16px 0; border-radius:6px; font-size:1.1rem;">
    <b>Pay Debug Info:</b><br>
    Old Balance: <?php echo number_format($_SESSION['pay_debug']['old_balance'], 2) . '<br>'; ?>
    Paid Amount: <?php echo number_format($_SESSION['pay_debug']['paid_amount'], 2) . '<br>'; ?>
    New Balance: <?php echo number_format($_SESSION['pay_debug']['new_balance'], 2) . '<br>'; ?>
    <?php if (!empty($_SESSION['pay_debug']['pay_debug_details'])): ?>
    <b>Details:</b><ul>
        <?php foreach ($_SESSION['pay_debug']['pay_debug_details'] as $detail): ?>
        <li><?php echo htmlspecialchars($detail); ?></li>
        <?php endforeach; ?>
    </ul>
    <?php endif; ?>
</div>
<?php unset($_SESSION['pay_debug']); endif; ?>

<!-- Add Reminder Modal -->
<div class="modal fade" id="reminderModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Send Reminder</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="reminderForm" method="POST">
                    <input type="hidden" name="reminder_customer_id" id="reminderCustomerId">
                    <div class="mb-3">
                        <label class="form-label">Customer</label>
                        <input type="text" class="form-control" id="reminderCustomerName" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Reminder Type</label>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="reminder_type[]" value="sms" id="smsReminder" checked <?php echo ($settings['enable_sms'] ?? '0') !== '1' ? 'disabled' : ''; ?>>
                            <label class="form-check-label" for="smsReminder">
                                SMS Reminder
                            </label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="reminder_type[]" value="email" id="emailReminder" checked>
                            <label class="form-check-label" for="emailReminder">
                                Email Reminder
                            </label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="reminder_type[]" value="whatsapp" id="whatsappReminder" checked <?php echo ($settings['enable_whatsapp'] ?? '0') !== '1' ? 'disabled' : ''; ?>>
                            <label class="form-check-label" for="whatsappReminder">
                                WhatsApp Reminder
                            </label>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Message</label>
                        <textarea class="form-control" name="reminder_message" id="reminderMessage" rows="3" required></textarea>
                        <small class="text-muted">You can use {customer_name}, {balance}, {shop_name}, and {shop_address} as placeholders.</small>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="submit" form="reminderForm" class="btn btn-primary" data-permission="send_reminder" data-permission-label="Send Reminder">Send Reminder</button>
            </div>
        </div>
    </div>
</div>

<script>
// ... existing code ...

// Add reminder button click handler
document.querySelectorAll('.reminder-btn').forEach(function(btn) {
    btn.addEventListener('click', function() {
        var cid = this.getAttribute('data-customer-id');
        var cname = this.getAttribute('data-customer-name');
        var balance = this.getAttribute('data-balance');
        var shopName = '<?php echo addslashes($storeName); ?>';
        var shopAddress = '<?php echo addslashes($storeAddress); ?>';
        var shopPhone = '<?php echo addslashes($storePhone); ?>';
        var currency = '<?php echo $currency; ?>';

        document.getElementById('reminderCustomerId').value = cid;
        document.getElementById('reminderCustomerName').value = cname;

        // Show loading in message and items preview
        document.getElementById('reminderMessage').value = 'Loading unpaid items...';
        let itemsPreview = document.getElementById('reminderItemsPreview');
        if (!itemsPreview) {
            itemsPreview = document.createElement('div');
            itemsPreview.id = 'reminderItemsPreview';
            itemsPreview.className = 'alert alert-info';
            document.getElementById('reminderMessage').parentNode.insertBefore(itemsPreview, document.getElementById('reminderMessage'));
        }
        itemsPreview.innerHTML = 'Loading unpaid items...';
        itemsPreview.style.display = 'block';

        // Fetch unpaid items
        fetch('../api/get_unpaid_items.php?customer_id=' + encodeURIComponent(cid))
            .then(response => response.json())
            .then(data => {
                let defaultMessage = `Dear {customer_name},\n\nThis is a payment reminder for your outstanding balance of {balance} at {shop_name}.`;
                let itemsText = '';
                let previewList = '';
                
                if (data.success && data.items && data.items.length > 0) {
                    itemsText += '\n\n📋 UNPAID INVOICE DETAILS:\n';
                    itemsText += '═'.repeat(40) + '\n';
                    previewList = '<div style="margin-bottom:0"><h6>Unpaid Invoice Details:</h6>';
                    
                    data.items.forEach(function(invoice, invoiceIndex) {
                        const invoiceDate = new Date(invoice.sale_date).toLocaleDateString();
                        const invoiceBalance = parseFloat(invoice.balance);
                        const totalPaidAmount = parseFloat(invoice.total_paid_amount) || 0;
                        const creditPaidAmount = parseFloat(invoice.credit_paid_amount) || parseFloat(invoice.paid_amount) || 0;
                        const posPaidAmount = parseFloat(invoice.pos_paid_amount) || 0;
                        const creditAmount = parseFloat(invoice.credit_amount) || parseFloat(invoice.amount) || 0;
                        const waivedAmount = parseFloat(invoice.waived_amount) || 0;
                        
                        itemsText += `\n🧾 Invoice: INV-${invoice.invoice_number ? invoice.invoice_number.replace(/[^\d]/g, '') : (invoice.invoice || '').replace(/[^\d]/g, '')}\n`;
                        itemsText += `📅 Date: ${invoiceDate}\n`;
                        
                        previewList += `<div class="border p-2 mb-2">`;
                        previewList += `<strong>INV-${invoice.invoice_number ? invoice.invoice_number.replace(/[^\d]/g, '') : (invoice.invoice || '').replace(/[^\d]/g, '')}</strong> (${invoiceDate})<br>`;
                        
                        if (invoice.items && invoice.items.length > 0) {
                            itemsText += '📦 Items:\n';
                            previewList += '<small><strong>Items:</strong><br>';
                            
                            invoice.items.forEach(function(item, itemIndex) {
                                const itemTotal = parseFloat(item.quantity) * parseFloat(item.price);
                                const serialInfo = item.serial_numbers ? ` [SN: ${item.serial_numbers}]` : '';
                                itemsText += `   ${itemIndex + 1}. ${item.item_name}${serialInfo}\n`;
                                itemsText += `      Qty: ${item.quantity} × ${currency} ${parseFloat(item.price).toFixed(2)} = ${currency} ${itemTotal.toFixed(2)}\n`;
                                
                                previewList += `${itemIndex + 1}. ${item.item_name}${serialInfo} (Qty: ${item.quantity} × ${currency} ${parseFloat(item.price).toFixed(2)} = ${currency} ${itemTotal.toFixed(2)})<br>`;
                            });
                            previewList += '</small>';
                        }
                        
                        itemsText += `💰 Credit Amount: ${currency} ${creditAmount.toFixed(2)}\n`;
                        
                        // Show detailed payment breakdown
                        if (posPaidAmount > 0) {
                            if (invoice.pos_payments && invoice.pos_payments.length > 0) {
                                if (invoice.pos_payments.length === 1) {
                                    // Single payment method
                                    const payment = invoice.pos_payments[0];
                                    itemsText += `💳 POS Paid (${payment.method.toUpperCase()}): ${currency} ${posPaidAmount.toFixed(2)}\n`;
                                } else {
                                    // Multiple payment methods
                                    itemsText += `💳 POS Paid: ${currency} ${posPaidAmount.toFixed(2)}\n`;
                                    invoice.pos_payments.forEach(function(payment, index) {
                                        itemsText += `   ${payment.method.toUpperCase()}: ${currency} ${parseFloat(payment.amount).toFixed(2)}\n`;
                                    });
                                }
                            } else {
                                itemsText += `💳 POS Paid: ${currency} ${posPaidAmount.toFixed(2)}\n`;
                            }
                        }
                        
                        if (creditPaidAmount > 0) {
                            itemsText += `💳 Credit Paid: ${currency} ${creditPaidAmount.toFixed(2)}\n`;
                        }
                        
                        if (waivedAmount > 0) {
                            itemsText += `🏷️ Amount Waived: ${currency} ${waivedAmount.toFixed(2)}\n`;
                        }
                        
                        itemsText += `⚠️ Balance: ${currency} ${invoiceBalance.toFixed(2)}\n`;
                        
                        previewList += `Credit: <span class="text-primary">${currency} ${creditAmount.toFixed(2)}</span>`;
                        if (posPaidAmount > 0) {
                            previewList += ` <small>(Note: POS payments of ${currency} ${posPaidAmount.toFixed(2)} were for original sale)</small>`;
                        }
                        if (waivedAmount > 0) {
                            previewList += ` | <span class="text-warning">Waived: ${currency} ${waivedAmount.toFixed(2)}</span>`;
                        }
                        previewList += ` | <strong>Balance: <span class="text-danger">${currency} ${invoiceBalance.toFixed(2)}</span></strong><br>`;
                        previewList += `</div>`;
                        
                        if (invoiceIndex < data.items.length - 1) {
                            itemsText += '\n' + '─'.repeat(40) + '\n';
                        }
                    });
                    
                    itemsText += '\n' + '═'.repeat(40);
                    previewList += '</div>';
                    itemsPreview.innerHTML = previewList;
                    
                    // Calculate totals for summary
                    let totalCredit = 0;
                    let totalPaid = 0;
                    let totalWaived = 0;
                    let totalOutstanding = 0;
                    
                    data.items.forEach(function(invoice) {
                        const creditAmount = parseFloat(invoice.credit_amount) || parseFloat(invoice.amount) || 0;
                        const creditPaidAmount = parseFloat(invoice.credit_paid_amount) || parseFloat(invoice.paid_amount) || 0;
                        const posPaidAmount = parseFloat(invoice.pos_paid_amount) || 0;
                        const waivedAmount = parseFloat(invoice.waived_amount) || 0;
                        const balance = parseFloat(invoice.balance) || 0;
                        
                        totalCredit += creditAmount;
                        totalPaid += creditPaidAmount; // Only actual payments toward credit debt
                        totalWaived += waivedAmount;
                        totalOutstanding += balance;
                    });
                    
                    // Add summary to message
                    itemsText += `\n\n💰 SUMMARY:\n`;
                    itemsText += '═'.repeat(30) + '\n';
                    if (totalWaived > 0) {
                        itemsText += `🏷️ Total Waived: ${currency} ${totalWaived.toFixed(2)}\n`;
                    }
                    itemsText += `⚠️ Total Outstanding Debt: ${currency} ${totalOutstanding.toFixed(2)}`;
                    
                    // Add summary to preview
                    previewList = previewList.replace('</div>', 
                        `<div class="border-top pt-2 mt-2">
                            <strong>Summary:</strong><br>
                            ${totalWaived > 0 ? `Total Waived: <span class="text-warning">${currency} ${totalWaived.toFixed(2)}</span><br>` : ''}
                            <strong>Total Outstanding Debt: <span class="text-danger">${currency} ${totalOutstanding.toFixed(2)}</span></strong>
                        </div>
                    </div>`);
                    itemsPreview.innerHTML = previewList;
                } else {
                    itemsPreview.innerHTML = '<div class="alert alert-warning"><b>No unpaid items found for this customer.</b></div>';
                }
                
                defaultMessage += itemsText;
                defaultMessage += `\n\n📞 Contact us on ${shopPhone} for payment methods.`;
                defaultMessage += `\n📍 Visit us at: {shop_address}`;
                defaultMessage += `\n\nThank you for your business!\n{shop_name}`;
                
                defaultMessage = defaultMessage
                    .replace(/{customer_name}/g, cname)
                    .replace(/{balance}/g, currency + ' ' + balance)
                    .replace(/{shop_name}/g, shopName)
                    .replace(/{shop_address}/g, shopAddress);
                document.getElementById('reminderMessage').value = defaultMessage;
            })
            .catch(() => {
                itemsPreview.innerHTML = '<b>Could not load unpaid items.</b>';
                document.getElementById('reminderMessage').value = `Dear ${cname}, this is a reminder that you have a pending payment of ${currency} ${balance} at ${shopName}, located at ${shopAddress}. 📞 Contact us on ${shopPhone} for payment methods.`;
            });

        var modal = new bootstrap.Modal(document.getElementById('reminderModal'));
        modal.show();
    });
});

// Handle reminder form submission
document.getElementById('reminderForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    // Get the submit button from the modal footer
    var submitBtn = document.querySelector('#reminderModal .modal-footer button[type="submit"]');
    if (!submitBtn || submitBtn.disabled) return;
    
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';

    var formData = new FormData(this);
    var message = formData.get('reminder_message'); // Already filled with real info

    // Send reminders based on selected types
    var reminderTypes = formData.getAll('reminder_type[]');
    var promises = [];

    reminderTypes.forEach(function(type) {
        if (type === 'sms') {
            promises.push(
                fetch('../helpers/send_reminder.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        type: 'sms',
                        customer_id: formData.get('reminder_customer_id'),
                        message: message
                    })
                }).then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
            );
        } else if (type === 'email') {
            promises.push(
                fetch('../helpers/send_reminder.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        type: 'email',
                        customer_id: formData.get('reminder_customer_id'),
                        message: message
                    })
                }).then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
            );
        } else if (type === 'whatsapp') {
            promises.push(
                fetch('../helpers/send_reminder.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        type: 'whatsapp',
                        customer_id: formData.get('reminder_customer_id'),
                        message: message
                    })
                }).then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                }).then(result => {
                    // Handle WhatsApp link generation
                    console.log('WhatsApp result:', result); // Debug log
                    if (result.success && result.whatsapp_url) {
                        // Add a small delay to ensure this dialog appears after any loading states
                        setTimeout(() => {
                            if (confirm('✅ WhatsApp reminder ready!\n\nClick OK to open WhatsApp and send the reminder message.')) {
                                openWhatsAppWithFallback(result.whatsapp_url, result.alternative_urls);
                            }
                        }, 100);
                    }
                    return result;
                })
            );
        }
    });

    Promise.all(promises)
        .then(results => {
            var success = results.every(r => r.success);
            var message = '';
            var hasWhatsAppUrl = false;
            
            // Check if any result has a WhatsApp URL (means it needs manual sending)
            results.forEach(result => {
                if (result.whatsapp_url) {
                    hasWhatsAppUrl = true;
                }
            });
            
            if (success) {
                // If there's a WhatsApp URL, modify the success message
                if (hasWhatsAppUrl) {
                    message = 'Reminders processed successfully! WhatsApp link opened for manual sending.';
                } else {
                    message = 'Reminders sent successfully!';
                }
                var modal = bootstrap.Modal.getInstance(document.getElementById('reminderModal'));
                modal.hide();
            } else {
                var failedTypes = [];
                results.forEach((result, index) => {
                    if (!result.success) {
                        failedTypes.push(reminderTypes[index]);
                    }
                });
                message = 'Failed to send ' + failedTypes.join(' and ') + ' reminder(s). Please try again.';
            }
            
            // Show notification modal
            showReminderNotification(message, success);
        })
        .catch(error => {
            console.error('Error:', error);
            showReminderNotification('An error occurred while sending reminders. Please try again.', false);
        })
        .finally(() => {
            if (submitBtn) {
                submitBtn.disabled = false;
                submitBtn.innerHTML = 'Send Reminder';
            }
        });
});

// ... existing code ...

// WhatsApp/SMS toggle logic
function updateReminderTypeToggles() {
    var smsEnabled = <?php echo ($settings['enable_sms'] ?? '0') === '1' ? 'true' : 'false'; ?>;
    var whatsappEnabled = <?php echo ($settings['enable_whatsapp'] ?? '0') === '1' ? 'true' : 'false'; ?>;
    document.getElementById('smsReminder').disabled = !smsEnabled;
    document.getElementById('whatsappReminder').disabled = !whatsappEnabled;
}

// Enhanced WhatsApp opener with fallback URLs for better app compatibility
function openWhatsAppWithFallback(primaryUrl, alternativeUrls) {
    // Try to detect if user is on mobile device
    const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
    
    if (isMobile && alternativeUrls && alternativeUrls.app_url) {
        // On mobile, first try the whatsapp:// protocol to open the app directly
        const appUrl = alternativeUrls.app_url;
        
        // Create a hidden iframe to try the app URL
        const iframe = document.createElement('iframe');
        iframe.style.display = 'none';
        iframe.src = appUrl;
        document.body.appendChild(iframe);
        
        // Set a timeout to fallback to web URL if app doesn't open
        setTimeout(() => {
            document.body.removeChild(iframe);
            // If app didn't open, try the primary URL
            window.open(primaryUrl, '_blank');
        }, 1000);
        
        // Also try to open the app URL directly
        window.location.href = appUrl;
        
        // Fallback: if nothing works after 2 seconds, open web version
        setTimeout(() => {
            if (alternativeUrls.web_url) {
                window.open(alternativeUrls.web_url, '_blank');
            }
        }, 2000);
    } else {
        // On desktop or if no alternative URLs, use primary URL
        window.open(primaryUrl, '_blank');
        
        // Also provide web.whatsapp.com option for desktop users
        if (alternativeUrls && alternativeUrls.web_url) {
            setTimeout(() => {
                if (confirm('If WhatsApp didn\'t open, would you like to try WhatsApp Web?')) {
                    window.open(alternativeUrls.web_url, '_blank');
                }
            }, 3000);
        }
    }
}

// Show reminder notification function
function showReminderNotification(message, isSuccess) {
    // Create notification element
    var notification = document.createElement('div');
    notification.className = 'alert alert-' + (isSuccess ? 'success' : 'danger') + ' alert-dismissible fade show position-fixed';
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    // Add to body
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(function() {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}

document.addEventListener('DOMContentLoaded', updateReminderTypeToggles);
</script>

<!-- Unpaid Receipt Templates -->
<?php foreach ($credits_by_customer as $cid => $group): ?>
<template id="unpaid-receipt-template-<?php echo $cid; ?>" data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" data-total-balance="<?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?>">
    <div style="width:80mm; font-family: Arial, Helvetica, sans-serif; font-size:13px; color:#111; font-weight:700;">
        <div style="text-align:center; margin-bottom:10px; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <div style="font-size:1.2rem; font-weight:700; color:#111; margin-bottom:4px; letter-spacing:1px; font-family: Arial, Helvetica, sans-serif;">
                <h1 style="font-size:1.2rem; font-weight:700; color:#111; margin-bottom:4px; letter-spacing:1px; font-family: Arial, Helvetica, sans-serif;">UNPAID CREDIT ITEMS</h1>
                <h1 style="font-size:1.2rem; font-weight:700; color:#111; margin-bottom:4px; letter-spacing:1px; font-family: Arial, Helvetica, sans-serif;">RECEIPT</h1>
            </div>
            <?php if ($logoData): ?>
                <?php
                $mimeType = 'image/jpeg'; // Default to JPEG
                if (strpos(strtolower($logoName), '.png') !== false) {
                    $mimeType = 'image/png';
                } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                    $mimeType = 'image/gif';
                }
                ?>
                <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
            <?php endif; ?>
        </div>
        <div style="text-align:center; margin-bottom:10px; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <div style="font-size:1.1rem; font-weight:700; color:#111; font-family: Arial, Helvetica, sans-serif;">POS</div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($storeAddress); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Phone: <?php echo htmlspecialchars($storePhone); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Email: <?php echo htmlspecialchars($storeEmail); ?></div>
            <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;">Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
        </div>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Customer:</b> <?php echo htmlspecialchars($group['customer_name']); ?></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Date:</b> <span class="receipt-date"></span></div>
        <div style="font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Receipt No:</b> <span class="receipt-number"></span></div>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-weight:700; margin-bottom:4px; font-family: Arial, Helvetica, sans-serif;">Unpaid Credit Items:</div>
        <table style="width:100%; font-size:12px; border-collapse:collapse; margin-bottom:10px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
            <thead>
                <tr style="background:#f1f5f9; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Date</th>
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Invoice</th>
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Items</th>
                    <th style="text-align:center; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Qty</th>
                    <th style="text-align:left; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Payment Details</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Waived</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Credit Amount</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Paid</th>
                    <th style="text-align:right; padding:6px 4px; font-size:13px; font-weight:700; border-bottom:1px solid #000; color:#111; font-family: Arial, Helvetica, sans-serif;">Balance</th>
                </tr>
            </thead>
            <tbody>
                <?php $rowAlt = false; foreach ($group['transactions'] as $credit): ?>
                <?php $item_balance = $credit['amount'] - $credit['amount_paid']; if ($item_balance <= 0.001) continue; ?>
                <tr style="background:<?php echo $rowAlt ? '#f8fafc' : '#fff'; ?>; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo date('M d', strtotime($credit['sale_date'])); ?></td>
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($credit['invoice_number']); ?></td>
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo htmlspecialchars($credit['items'] ?? 'No items found'); ?></td>
                    <td style="text-align:center; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $credit['total_quantity'] ?? 0; ?></td>
                    <td style="padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">
                        <?php
                        $paymentDetails = '';
                        if (!empty($credit['payment_methods'])) {
                            $methods = explode(', ', $credit['payment_methods']);
                            $amounts = explode(', ', $credit['payment_amounts']);
                            $details = [];
                            for ($i = 0; $i < count($methods); $i++) {
                                if (isset($amounts[$i]) && $amounts[$i] > 0) {
                                    $details[] = ucfirst($methods[$i]) . ': ' . $currency . ' ' . number_format($amounts[$i], 2);
                                }
                            }
                            $paymentDetails = implode(', ', $details);
                        } elseif (!empty($credit['sale_payment_method']) && $credit['sale_paid_amount'] > 0) {
                            $paymentDetails = ucfirst($credit['sale_payment_method']) . ': ' . $currency . ' ' . number_format($credit['sale_paid_amount'], 2);
                        } else {
                            $paymentDetails = 'Credit Sale';
                        }
                        echo $paymentDetails;
                        ?>
                    </td>
                    <td style="text-align:right; padding:5px 4px; color:#dc2626; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $waived > 0 ? ($currency . ' ' . number_format($waived, 2)) : '-'; ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount'], 2); ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($credit['amount_paid'], 2); ?></td>
                    <td style="text-align:right; padding:5px 4px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><?php echo $currency . ' ' . number_format($item_balance, 2); ?></td>
                </tr>
                <?php $rowAlt = !$rowAlt; endforeach; ?>
            </tbody>
        </table>
        <div style="border-top:1px dashed #000; margin:10px 0;"></div>
        <div style="font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;"><b>Total Balance:</b> <?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?></div>
        <div style="margin:14px 0 6px 0; text-align:center; font-size:13px; color:#111; font-family: Arial, Helvetica, sans-serif; font-weight:700;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
    </div>
</template>
<?php endforeach; ?>

<!-- Add Credit Item Modal -->
<div class="modal fade" id="addCreditItemModal" tabindex="-1">
  <div class="modal-dialog modal-lg">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Add Credit Items for <span id="addCreditCustomerName"></span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <form id="addCreditItemForm" method="POST">
        <input type="hidden" name="add_credit_customer_id" id="addCreditCustomerId">
        <div class="modal-body">
          <div id="creditItemsContainer">
            <!-- Dynamic product rows will be added here -->
          </div>
          <div class="text-center mt-3">
            <button type="button" class="btn btn-outline-primary" id="addCreditItemRow">
              <i class="fas fa-plus"></i> Add Another Item
            </button>
          </div>
          <hr>
          <div class="mb-3">
            <label class="form-label">Reminder Type</label>
            <div id="reminderTypeCheckboxes" class="d-flex gap-3 flex-wrap">
              <div class="form-check">
                <input class="form-check-input" type="checkbox" name="reminder_type[]" value="sms" id="reminderTypeSms" 
                       <?php echo ($settings['enable_sms'] ?? '0') === '1' ? 'checked' : 'disabled'; ?>>
                <label class="form-check-label <?php echo ($settings['enable_sms'] ?? '0') === '1' ? '' : 'text-muted'; ?>" for="reminderTypeSms">
                  SMS
                  <?php if (($settings['enable_sms'] ?? '0') !== '1'): ?>
                    <small class="text-muted d-block">(Disabled in settings)</small>
                  <?php endif; ?>
                </label>
              </div>
              <div class="form-check">
                <input class="form-check-input" type="checkbox" name="reminder_type[]" value="email" id="reminderTypeEmail" 
                       <?php echo ($settings['enable_email'] ?? '0') === '1' ? 'checked' : 'disabled'; ?>>
                <label class="form-check-label <?php echo ($settings['enable_email'] ?? '0') === '1' ? '' : 'text-muted'; ?>" for="reminderTypeEmail">
                  Email
                  <?php if (($settings['enable_email'] ?? '0') !== '1'): ?>
                    <small class="text-muted d-block">(Disabled in settings)</small>
                  <?php endif; ?>
                </label>
              </div>
              <div class="form-check">
                <input class="form-check-input" type="checkbox" name="reminder_type[]" value="whatsapp" id="reminderTypeWhatsapp" 
                       <?php echo ($settings['enable_whatsapp'] ?? '0') === '1' ? 'checked' : 'disabled'; ?>>
                <label class="form-check-label <?php echo ($settings['enable_whatsapp'] ?? '0') === '1' ? '' : 'text-muted'; ?>" for="reminderTypeWhatsapp">
                  WhatsApp
                  <?php if (($settings['enable_whatsapp'] ?? '0') !== '1'): ?>
                    <small class="text-muted d-block">(Disabled in settings)</small>
                  <?php endif; ?>
                </label>
              </div>
            </div>
            <small class="text-muted">Only enabled notification methods will be available. You can enable/disable these in Settings.</small>
          </div>
          <div class="mb-3">
            <label class="form-label">Message Preview</label>
            <div id="creditMessagePreview" class="alert alert-info" style="white-space:pre-line;"></div>
          </div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
          <button type="submit" class="btn btn-dark" data-permission="add_credit_item" data-permission-label="Add Credit Item">Add Credit Items</button>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- Serial Number Selection Modal -->
<div class="modal fade" id="serialNumberModal" tabindex="-1">
  <div class="modal-dialog modal-lg">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Select Serial Numbers for <span id="serialProductName"></span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <div class="alert alert-info">
          <i class="fas fa-info-circle"></i>
          Please select <span id="requiredSerialCount">0</span> serial number(s) for this product.
        </div>
        <div id="serialNumbersContainer" class="row">
          <!-- Serial number checkboxes will be populated here -->
        </div>
        <div class="mt-3">
          <small class="text-muted">
            <i class="fas fa-check-circle text-success"></i> Selected: <span id="selectedSerialCount">0</span>
          </small>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-primary" id="confirmSerialSelection">Confirm Selection</button>
      </div>
    </div>
  </div>
</div>

<style>
.credit-item-row {
  border: 1px solid #dee2e6;
  border-radius: 8px;
  padding: 15px;
  margin-bottom: 15px;
  background-color: #f8f9fa;
}

.credit-item-row .row {
  align-items: end;
}

.remove-item-btn {
  color: #dc3545;
  border: none;
  background: none;
  font-size: 1.2rem;
  padding: 0;
  line-height: 1;
}

.remove-item-btn:hover {
  color: #c82333;
}

.item-total {
  font-weight: bold;
  color: #495057;
  background-color: #e9ecef;
  padding: 8px 12px;
  border-radius: 4px;
  text-align: center;
}

/* Serial number modal styles */
.serial-checkbox {
  margin-right: 8px;
}

.serial-checkbox:checked + label {
  font-weight: bold;
  color: #0d6efd;
}

.selected-serials-display {
  background-color: #e9ecef;
  padding: 4px 8px;
  border-radius: 4px;
  border: 1px solid #ced4da;
  word-break: break-all;
}

.select-serials-btn {
  font-size: 0.875rem;
}

.no-serials-message {
  font-style: italic;
}

/* Price validation styles */
.price-input.is-invalid {
  border-color: #dc3545;
  box-shadow: 0 0 0 0.2rem rgba(220, 53, 69, 0.25);
}

.price-input.is-invalid:focus {
  border-color: #dc3545;
  box-shadow: 0 0 0 0.2rem rgba(220, 53, 69, 0.25);
}
</style>

<script>
// Helper: get product name by id
const productMap = {
  <?php foreach ($products as $product): ?>
    '<?php echo $product['id']; ?>': '<?php echo addslashes($product['name']); ?>',
  <?php endforeach; ?>
};

function buildCreditMessagePreview() {
  const customerName = document.getElementById('addCreditCustomerName').textContent;
  const customerId = document.getElementById('addCreditCustomerId').value;
  const currency = '<?php echo $currency; ?>';
  const storeName = '<?php echo addslashes($storeName); ?>';
  const storePhone = '<?php echo addslashes($storePhone); ?>';
  const storeAddress = '<?php echo addslashes($storeAddress); ?>';
  let items = [];
  let totalAmount = 0;
  
  document.querySelectorAll('#creditItemsContainer .credit-item-row').forEach(row => {
    const productId = row.querySelector('.product-select').value;
    const quantity = row.querySelector('.quantity-input').value;
    const price = row.querySelector('.price-input').value;
    const serialNumbersInput = row.querySelector('.serial-numbers-input');
    
    if (productId && quantity && price) {
      const productName = productMap[productId] || '';
      const total = parseFloat(quantity) * parseFloat(price);
      totalAmount += total;
      
      let itemText = `${productName} (Qty: ${quantity} @ ${currency} ${parseFloat(price).toFixed(2)})`;
      
      // Add serial numbers if available
      if (serialNumbersInput && serialNumbersInput.value) {
        try {
          const serialNumbers = JSON.parse(serialNumbersInput.value);
          if (serialNumbers && serialNumbers.length > 0) {
            itemText += ` - SN: ${serialNumbers.join(', ')}`;
          }
        } catch (e) {
          // Ignore JSON parse errors
        }
      }
      
      items.push(itemText);
    }
  });
  
  let itemsList = items.join(', ');
  if (itemsList === '') {
    itemsList = 'No items selected';
  }
  
  // Get current balance from the table row for this customer
  let currentBalance = 0;
  const customerRow = document.querySelector(`tr[data-customer-id="${customerId}"]`);
  if (customerRow) {
    const balanceCell = customerRow.querySelector('td:nth-child(5)'); // Total Balance column
    if (balanceCell) {
      const balanceText = balanceCell.textContent.trim();
      const balanceMatch = balanceText.match(new RegExp(`${currency.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\s*([\\d,]+(?:\\.\\d{2})?)`));
      if (balanceMatch) {
        currentBalance = parseFloat(balanceMatch[1].replace(/,/g, ''));
      }
    }
  }
  
  // Calculate new total balance (current + new items)
  const newTotalBalance = currentBalance + totalAmount;
  
  // Build the message with store name, phone, and address
  let msg = `Dear ${customerName}, your credit items have been added: ${itemsList}. Total: ${currency} ${totalAmount.toFixed(2)}. Your total outstanding Debt balance is now ${currency} ${newTotalBalance.toFixed(2)}. Please visit ${storeName} at ${storeAddress} to make payment or contact us on ${storePhone} for payment methods. Thank you!`;
  
  const previewElement = document.getElementById('creditMessagePreview');
  if (previewElement) {
    previewElement.textContent = msg;
  }
}

// Function to create a new credit item row
function createCreditItemRow() {
  const container = document.getElementById('creditItemsContainer');
  const rowIndex = container.children.length;
  
  const row = document.createElement('div');
  row.className = 'credit-item-row';
  row.innerHTML = `
    <div class="row">
      <div class="col-md-3">
            <label class="form-label">Product</label>
        <select name="items[${rowIndex}][product_id]" class="form-select product-select" required>
              <option value="">Select Product</option>
              <?php foreach ($products as $product): ?>
                <option value="<?php echo $product['id']; ?>" 
                        data-price="<?php echo $product['price']; ?>"
                        data-cost-price="<?php echo $product['cost_price']; ?>"
                        data-require-serials="<?php echo $product['require_serials'] ?? 0; ?>"
                        data-category="<?php echo htmlspecialchars($product['category_name'] ?? ''); ?>">
                  <?php echo htmlspecialchars($product['name']); ?> (<?php echo $currency . ' ' . number_format($product['price'], 2); ?>)
                </option>
              <?php endforeach; ?>
            </select>
          </div>
      <div class="col-md-2">
        <label class="form-label">Serial No:</label>
        <div class="serial-numbers-section" style="display: none;">
          <button type="button" class="btn btn-outline-secondary btn-sm select-serials-btn" style="width: 100%;">
            <i class="fas fa-barcode"></i> Select Serials
          </button>
          <input type="hidden" name="items[${rowIndex}][serial_numbers]" class="serial-numbers-input">
          <div class="selected-serials-display mt-1" style="font-size: 0.8em; color: #6c757d;"></div>
        </div>
        <div class="no-serials-message" style="display: block; font-size: 0.8em; color: #6c757d;">
          <i class="fas fa-info-circle"></i> No serials required
        </div>
      </div>
      <div class="col-md-2">
            <label class="form-label">Quantity</label>
        <input type="number" name="items[${rowIndex}][quantity]" class="form-control quantity-input" min="1" value="1" required>
          </div>
      <div class="col-md-2">
            <label class="form-label">Price</label>
        <input type="number" name="items[${rowIndex}][price]" class="form-control price-input" min="0" step="0.01" required>
          </div>
      <div class="col-md-2">
        <label class="form-label">Total</label>
        <div class="item-total"><?php echo $currency; ?> 0.00</div>
        </div>
      <div class="col-md-1">
        <label class="form-label">&nbsp;</label>
        <button type="button" class="btn remove-item-btn" onclick="removeCreditItemRow(this)">
          <i class="fas fa-times"></i>
        </button>
        </div>
    </div>
  `;
  
  container.appendChild(row);
  
  // Add event listeners to the new row
  const productSelect = row.querySelector('.product-select');
  const quantityInput = row.querySelector('.quantity-input');
  const priceInput = row.querySelector('.price-input');
  const serialNumbersSection = row.querySelector('.serial-numbers-section');
  const noSerialsMessage = row.querySelector('.no-serials-message');
  const selectSerialsBtn = row.querySelector('.select-serials-btn');
  const serialNumbersInput = row.querySelector('.serial-numbers-input');
  const selectedSerialsDisplay = row.querySelector('.selected-serials-display');
  
  productSelect.addEventListener('change', function() {
    const selectedOption = this.options[this.selectedIndex];
    const price = selectedOption.getAttribute('data-price');
    const costPrice = selectedOption.getAttribute('data-cost-price');
    const requireSerials = selectedOption.getAttribute('data-require-serials') === '1';
    
    priceInput.value = price || '';
    
    // Set minimum price to cost price
    if (costPrice && costPrice > 0) {
      priceInput.min = costPrice;
      priceInput.placeholder = `Min: ${costPrice}`;
    } else {
      priceInput.min = 0;
      priceInput.placeholder = '';
    }
    
    // Show/hide serial numbers section
    if (requireSerials) {
      serialNumbersSection.style.display = 'block';
      noSerialsMessage.style.display = 'none';
    } else {
      serialNumbersSection.style.display = 'none';
      noSerialsMessage.style.display = 'block';
      serialNumbersInput.value = '';
      selectedSerialsDisplay.textContent = '';
    }
    
    calculateItemTotal(row);
    buildCreditMessagePreview();
  });
  
  quantityInput.addEventListener('input', function() {
    calculateItemTotal(row);
    buildCreditMessagePreview();
  });
  
  priceInput.addEventListener('input', function() {
    const selectedOption = productSelect.options[productSelect.selectedIndex];
    const costPrice = parseFloat(selectedOption.getAttribute('data-cost-price')) || 0;
    const enteredPrice = parseFloat(this.value) || 0;
    
    // Validate price against cost price
    if (costPrice > 0 && enteredPrice < costPrice) {
      this.setCustomValidity(`Price cannot be less than cost price (${costPrice})`);
      this.classList.add('is-invalid');
    } else {
      this.setCustomValidity('');
      this.classList.remove('is-invalid');
    }
    
    calculateItemTotal(row);
    buildCreditMessagePreview();
  });
  
  // Add serial number selection event
  selectSerialsBtn.addEventListener('click', function() {
    const selectedOption = productSelect.options[productSelect.selectedIndex];
    const productId = selectedOption.value;
    const productName = selectedOption.textContent.split(' (')[0];
    const quantity = parseInt(quantityInput.value) || 0;
    
    if (!productId || quantity <= 0) {
      alert('Please select a product and enter a valid quantity first.');
      return;
    }
    
    showSerialNumberModal(productId, productName, quantity, row);
  });
  
  // Update preview after adding row
  setTimeout(buildCreditMessagePreview, 100);
}

// Function to calculate item total
function calculateItemTotal(row) {
  const quantity = parseFloat(row.querySelector('.quantity-input').value) || 0;
  const price = parseFloat(row.querySelector('.price-input').value) || 0;
  const total = quantity * price;
  row.querySelector('.item-total').textContent = '<?php echo $currency; ?> ' + total.toFixed(2);
}

// Function to remove a credit item row
function removeCreditItemRow(button) {
  const container = document.getElementById('creditItemsContainer');
  if (container.children.length > 1) {
    button.closest('.credit-item-row').remove();
    // Reindex the remaining rows
    const rows = container.querySelectorAll('.credit-item-row');
    rows.forEach((row, index) => {
      const inputs = row.querySelectorAll('input, select');
      inputs.forEach(input => {
        const name = input.name;
        if (name.includes('[product_id]')) {
          input.name = name.replace(/\[\d+\]/, `[${index}]`);
        } else if (name.includes('[quantity]')) {
          input.name = name.replace(/\[\d+\]/, `[${index}]`);
        } else if (name.includes('[price]')) {
          input.name = name.replace(/\[\d+\]/, `[${index}]`);
        } else if (name.includes('[serial_numbers]')) {
          input.name = name.replace(/\[\d+\]/, `[${index}]`);
        }
      });
    });
    
    // Update preview after removing row
    setTimeout(buildCreditMessagePreview, 100);
  }
}

// Initialize the modal
document.querySelectorAll('.add-credit-item-btn').forEach(function(btn) {
  btn.addEventListener('click', function() {
    var customerId = this.getAttribute('data-customer-id');
    var customerName = this.getAttribute('data-customer-name');
    var totalCredit = this.getAttribute('data-total-credit');
    var totalPaid = this.getAttribute('data-total-paid');
    var totalWaived = this.getAttribute('data-total-waived');
    var totalBalance = this.getAttribute('data-total-balance');
    
    document.getElementById('addCreditCustomerId').value = customerId;
    document.getElementById('addCreditCustomerName').textContent = customerName;
    
    // Store current balance data to use instead of recalculating
    document.getElementById('addCreditCustomerId').setAttribute('data-current-total-credit', totalCredit);
    document.getElementById('addCreditCustomerId').setAttribute('data-current-total-paid', totalPaid);
    document.getElementById('addCreditCustomerId').setAttribute('data-current-total-waived', totalWaived);
    document.getElementById('addCreditCustomerId').setAttribute('data-current-total-balance', totalBalance);
    
    // Clear and initialize the container
    const container = document.getElementById('creditItemsContainer');
    container.innerHTML = '';
    
    // Add the first row
    createCreditItemRow();
    
    // Setup reminder type change events
    document.querySelectorAll('input[name="reminder_type[]"]:not(:disabled)').forEach(cb => {
      cb.addEventListener('change', buildCreditMessagePreview);
    });
    
    var modal = new bootstrap.Modal(document.getElementById('addCreditItemModal'));
    modal.show();
  });
});

// Add row button
document.getElementById('addCreditItemRow').addEventListener('click', function() {
  createCreditItemRow();
});

// Serial number modal functionality
let currentSerialRow = null;

function showSerialNumberModal(productId, productName, quantity, row) {
  currentSerialRow = row;
  
  // Update modal content
  document.getElementById('serialProductName').textContent = productName;
  document.getElementById('requiredSerialCount').textContent = quantity;
  document.getElementById('selectedSerialCount').textContent = '0';
  
  // Clear previous content
  document.getElementById('serialNumbersContainer').innerHTML = '';
  
  // Show loading
  document.getElementById('serialNumbersContainer').innerHTML = '<div class="col-12 text-center"><i class="fas fa-spinner fa-spin"></i> Loading serial numbers...</div>';
  
  // Fetch available serial numbers
  fetch(`get_available_serials.php?product_id=${productId}`)
    .then(response => response.json())
    .then(data => {
      if (data.success && data.serial_numbers && data.serial_numbers.length > 0) {
        displaySerialNumbers(data.serial_numbers, quantity);
      } else {
        document.getElementById('serialNumbersContainer').innerHTML = 
          '<div class="col-12"><div class="alert alert-warning">No available serial numbers found for this product.</div></div>';
      }
    })
    .catch(error => {
      console.error('Error fetching serial numbers:', error);
      document.getElementById('serialNumbersContainer').innerHTML = 
        '<div class="col-12"><div class="alert alert-danger">Error loading serial numbers. Please try again.</div></div>';
    });
  
  // Show modal
  const modal = new bootstrap.Modal(document.getElementById('serialNumberModal'));
  modal.show();
}

function displaySerialNumbers(serialNumbers, requiredCount) {
  const container = document.getElementById('serialNumbersContainer');
  container.innerHTML = '';
  
  serialNumbers.forEach((serial, index) => {
    const col = document.createElement('div');
    col.className = 'col-md-4 col-sm-6 mb-2';
    col.innerHTML = `
      <div class="form-check">
        <input class="form-check-input serial-checkbox" type="checkbox" 
               value="${serial}" id="serial_${index}" 
               onchange="updateSerialSelection()">
        <label class="form-check-label" for="serial_${index}">
          ${serial}
        </label>
      </div>
    `;
    container.appendChild(col);
  });
  
  // Add select all/none buttons if there are many serials
  if (serialNumbers.length > 10) {
    const buttonRow = document.createElement('div');
    buttonRow.className = 'col-12 mb-3';
    buttonRow.innerHTML = `
      <div class="btn-group btn-group-sm">
        <button type="button" class="btn btn-outline-primary" onclick="selectAllSerials()">
          <i class="fas fa-check-double"></i> Select All
        </button>
        <button type="button" class="btn btn-outline-secondary" onclick="clearAllSerials()">
          <i class="fas fa-times"></i> Clear All
        </button>
      </div>
    `;
    container.insertBefore(buttonRow, container.firstChild);
  }
}

function updateSerialSelection() {
  const checkboxes = document.querySelectorAll('.serial-checkbox:checked');
  const selectedCount = checkboxes.length;
  const requiredCount = parseInt(document.getElementById('requiredSerialCount').textContent);
  
  document.getElementById('selectedSerialCount').textContent = selectedCount;
  
  // Enable/disable confirm button based on selection
  const confirmBtn = document.getElementById('confirmSerialSelection');
  if (selectedCount === requiredCount) {
    confirmBtn.disabled = false;
    confirmBtn.classList.remove('btn-secondary');
    confirmBtn.classList.add('btn-primary');
  } else {
    confirmBtn.disabled = true;
    confirmBtn.classList.remove('btn-primary');
    confirmBtn.classList.add('btn-secondary');
  }
}

function selectAllSerials() {
  const checkboxes = document.querySelectorAll('.serial-checkbox');
  const requiredCount = parseInt(document.getElementById('requiredSerialCount').textContent);
  
  checkboxes.forEach((checkbox, index) => {
    checkbox.checked = index < requiredCount;
  });
  
  updateSerialSelection();
}

function clearAllSerials() {
  const checkboxes = document.querySelectorAll('.serial-checkbox');
  checkboxes.forEach(checkbox => {
    checkbox.checked = false;
  });
  
  updateSerialSelection();
}

// Confirm serial selection
document.getElementById('confirmSerialSelection').addEventListener('click', function() {
  if (!currentSerialRow) return;
  
  const selectedSerials = Array.from(document.querySelectorAll('.serial-checkbox:checked'))
    .map(checkbox => checkbox.value);
  
  const serialNumbersInput = currentSerialRow.querySelector('.serial-numbers-input');
  const selectedSerialsDisplay = currentSerialRow.querySelector('.selected-serials-display');
  
  serialNumbersInput.value = JSON.stringify(selectedSerials);
  selectedSerialsDisplay.textContent = selectedSerials.join(', ');
  
  // Close modal
  const modal = bootstrap.Modal.getInstance(document.getElementById('serialNumberModal'));
  modal.hide();
  
  // Update message preview
  buildCreditMessagePreview();
});

// Form submission
document.getElementById('addCreditItemForm').addEventListener('submit', function(e) {
  e.preventDefault();
  
  // Validate that at least one item is selected
  const productSelects = document.querySelectorAll('.product-select');
  let hasValidItem = false;
  let validationErrors = [];
  
  productSelects.forEach((select, index) => {
    if (select.value) {
      hasValidItem = true;
      
      // Check if product requires serial numbers
      const selectedOption = select.options[select.selectedIndex];
      const requireSerials = selectedOption.getAttribute('data-require-serials') === '1';
      const quantity = parseInt(select.closest('.credit-item-row').querySelector('.quantity-input').value) || 0;
      
      if (requireSerials && quantity > 0) {
        const serialNumbersInput = select.closest('.credit-item-row').querySelector('.serial-numbers-input');
        let serialNumbers = [];
        
        try {
          if (serialNumbersInput.value) {
            serialNumbers = JSON.parse(serialNumbersInput.value);
          }
        } catch (e) {
          // Ignore JSON parse errors
        }
        
        if (!Array.isArray(serialNumbers) || serialNumbers.length !== quantity) {
          const productName = selectedOption.textContent.split(' (')[0];
          validationErrors.push(`Product "${productName}" requires ${quantity} serial number(s) to be selected.`);
        }
      }
    }
  });
  
  if (!hasValidItem) {
    showCreditErrorModal('Validation Error', 'Please select at least one product.', 'fas fa-shopping-cart');
    return;
  }
  
  if (validationErrors.length > 0) {
    showCreditErrorModal('Validation Error', validationErrors.join('\n'), 'fas fa-exclamation-triangle');
    return;
  }
  
  // Calculate total amount for credit validation
  let totalAmount = 0;
  document.querySelectorAll('#creditItemsContainer .credit-item-row').forEach(row => {
    const quantity = parseFloat(row.querySelector('.quantity-input').value || 0);
    const price = parseFloat(row.querySelector('.price-input').value || 0);
    totalAmount += quantity * price;
  });
  
  if (totalAmount <= 0) {
    showCreditErrorModal('Validation Error', 'Please enter valid quantities and prices for all items.', 'fas fa-calculator');
    return;
  }
  
  // Get customer ID for credit validation
  const customerId = document.getElementById('addCreditCustomerId').value;
  
  if (!customerId) {
    showCreditErrorModal('System Error', 'Customer ID is missing. Please try again.', 'fas fa-user-times');
    return;
  }
  
  var form = this;
  var formData = new FormData(form);
  
  // Add current balance data to form submission
  const customerIdField = document.getElementById('addCreditCustomerId');
  formData.append('current_total_credit', customerIdField.getAttribute('data-current-total-credit') || '0');
  formData.append('current_total_paid', customerIdField.getAttribute('data-current-total-paid') || '0');
  formData.append('current_total_waived', customerIdField.getAttribute('data-current-total-waived') || '0');
  formData.append('current_total_balance', customerIdField.getAttribute('data-current-total-balance') || '0');
  
  // Show loading state
  const submitBtn = form.querySelector('button[type="submit"]');
  const originalText = submitBtn.innerHTML;
  submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Validating credit...';
  submitBtn.disabled = true;
  
  // Validate customer credit before proceeding
  fetch(`../api/check_customer_credit.php?customer_id=${customerId}&sale_amount=${totalAmount}`)
    .then(response => response.json())
    .then(creditData => {
             if (!creditData.success) {
         // Credit validation failed - show error modal
         submitBtn.innerHTML = originalText;
         submitBtn.disabled = false;
         
         if (creditData.error === 'Customer is suspended and cannot make credit purchases') {
           showCreditErrorModal(
             'Customer Suspended',
             `<strong>${creditData.customer_name}</strong> is currently suspended and cannot receive credit items.`,
             'fas fa-user-slash'
           );
         } else if (creditData.error === 'Customer does not have credit enabled') {
           showCreditErrorModal(
             'Credit Not Enabled',
             `<strong>${creditData.customer_name}</strong> does not have credit enabled.<br><br>Current Credit Limit: <strong><?php echo $currency; ?> ${creditData.credit_limit.toFixed(2)}</strong>`,
             'fas fa-credit-card'
           );
         } else if (creditData.error === 'Sale amount exceeds available credit') {
           const creditSummaryHtml = `
             <strong>${creditData.customer_name}</strong> has insufficient credit for these items.
             <div class="mt-3">
               <div class="row text-start">
                 <div class="col-6"><strong>Credit Limit:</strong></div>
                 <div class="col-6"><?php echo $currency; ?> ${creditData.credit_limit.toFixed(2)}</div>
                 <div class="col-6"><strong>Current Usage:</strong></div>
                 <div class="col-6"><?php echo $currency; ?> ${creditData.current_usage.toFixed(2)}</div>
                 <div class="col-6"><strong>Available Credit:</strong></div>
                 <div class="col-6 text-success"><?php echo $currency; ?> ${creditData.available_credit.toFixed(2)}</div>
                 <div class="col-6"><strong>Items Total:</strong></div>
                 <div class="col-6 text-danger"><?php echo $currency; ?> ${totalAmount.toFixed(2)}</div>
                 <div class="col-6"><strong>Shortfall:</strong></div>
                 <div class="col-6 text-danger"><?php echo $currency; ?> ${creditData.deficit.toFixed(2)}</div>
               </div>
             </div>
           `;
           showCreditErrorModal('Insufficient Credit', creditSummaryHtml, 'fas fa-triangle-exclamation');
         } else {
           showCreditErrorModal('Credit Validation Error', creditData.error || 'Credit validation failed.', 'fas fa-exclamation-circle');
         }
         return;
       }
      
             // Credit validation passed - show warning if applicable and proceed
       if (creditData.warning_level !== 'none') {
         let warningTitle = '';
         let warningMessage = '';
         let warningIcon = 'fas fa-info-circle';
         let warningClass = 'warning';
         
         if (creditData.warning_level === 'critical') {
           warningTitle = 'High Credit Usage Warning';
           warningMessage = `Adding these credit items will bring <strong>${creditData.customer_name}'s</strong> credit usage to <strong>${creditData.usage_percent_after_sale.toFixed(1)}%</strong> of their limit.`;
           warningIcon = 'fas fa-exclamation-triangle';
           warningClass = 'danger';
         } else if (creditData.warning_level === 'warning') {
           warningTitle = 'Credit Usage Notice';
           warningMessage = `Adding these credit items will bring <strong>${creditData.customer_name}'s</strong> credit usage to <strong>${creditData.usage_percent_after_sale.toFixed(1)}%</strong> of their limit.`;
           warningIcon = 'fas fa-exclamation-triangle';
           warningClass = 'warning';
         } else if (creditData.warning_level === 'moderate') {
           warningTitle = 'Credit Information';
           warningMessage = `After adding these items, <strong>${creditData.customer_name}</strong> will have <strong><?php echo $currency; ?> ${creditData.available_after_sale.toFixed(2)}</strong> credit remaining.`;
           warningIcon = 'fas fa-info-circle';
           warningClass = 'info';
         }
         
         const creditSummaryHtml = `
           ${warningMessage}
           <div class="mt-3">
             <h6>Credit Summary:</h6>
             <div class="row text-start">
               <div class="col-6"><strong>Credit Limit:</strong></div>
               <div class="col-6"><?php echo $currency; ?> ${creditData.credit_limit.toFixed(2)}</div>
               <div class="col-6"><strong>Current Usage:</strong></div>
               <div class="col-6"><?php echo $currency; ?> ${creditData.current_usage.toFixed(2)}</div>
               <div class="col-6"><strong>Items Total:</strong></div>
               <div class="col-6 text-primary"><?php echo $currency; ?> ${totalAmount.toFixed(2)}</div>
               <div class="col-6"><strong>Available After:</strong></div>
               <div class="col-6 text-success"><?php echo $currency; ?> ${creditData.available_after_sale.toFixed(2)}</div>
             </div>
           </div>
         `;
         
         showCreditConfirmModal(
           warningTitle, 
           creditSummaryHtml, 
           warningIcon, 
           warningClass,
           function() {
             // User confirmed - proceed
             submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';
             processCreditItemsSubmission(form, formData, submitBtn, originalText);
           },
           function() {
             // User cancelled
             submitBtn.innerHTML = originalText;
             submitBtn.disabled = false;
           }
         );
       } else {
         // No warnings - proceed directly
         submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';
         processCreditItemsSubmission(form, formData, submitBtn, originalText);
       }
    })
         .catch(error => {
       console.error('Credit validation error:', error);
       submitBtn.innerHTML = originalText;
       submitBtn.disabled = false;
       showCreditErrorModal('Connection Error', 'Could not verify customer credit. Please check your connection and try again.', 'fas fa-wifi');
     });
});

// Function to process the credit items submission after validation
function processCreditItemsSubmission(form, formData, submitBtn, originalText) {
  
  fetch('', {
    method: 'POST',
    body: formData
  })
  .then(response => response.json())
  .then(data => {
    if (data.success) {
      // Handle WhatsApp URL if available
      if (data.whatsapp_url) {
        if (confirm(`✅ Credit items added successfully!\n\nWhatsApp message ready for ${data.customer_name}.\n\nClick OK to open WhatsApp and send the notification.`)) {
          openWhatsAppWithFallback(data.whatsapp_url, data.alternative_urls);
        }
      }
      
      // Parse the message to show detailed feedback
      const message = data.message;
      const successParts = message.split('.');
      const mainMessage = successParts[0]; // "Credit items added successfully"
      
      let messageStatus = '';
      let failedStatus = '';
      
      // Extract message details
      successParts.forEach(part => {
        part = part.trim();
        if (part.includes('sent successfully')) {
          messageStatus += part + '. ';
        } else if (part.startsWith('Failed:')) {
          failedStatus = part;
        }
      });
      
      // Create detailed success modal
      showDetailedSuccessModal(mainMessage, messageStatus, failedStatus);
      
      // Reload page after modal closes
      setTimeout(() => {
        location.reload();
      }, 3000);
    } else {
      alert('Error: ' + data.message);
    }
  })
  .catch(err => {
    console.error('Error:', err);
    alert('Error: ' + err);
  })
  .finally(() => {
    // Reset button state
    submitBtn.innerHTML = originalText;
    submitBtn.disabled = false;
  });
  
  var modal = bootstrap.Modal.getInstance(document.getElementById('addCreditItemModal'));
  modal.hide();
}

// Function to show credit error modal
function showCreditErrorModal(title, message, icon = 'fas fa-exclamation-circle') {
  // Remove existing modal if any
  const existingModal = document.getElementById('creditErrorModal');
  if (existingModal) {
    existingModal.remove();
  }
  
  // Create modal HTML
  const modalHTML = `
    <div class="modal fade" id="creditErrorModal" tabindex="-1" aria-labelledby="creditErrorModalLabel" aria-hidden="true">
      <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content" style="border: none; border-radius: 15px; box-shadow: 0 10px 30px rgba(0,0,0,0.2);">
          <div class="modal-body text-center p-4">
            <div class="mb-3">
              <i class="${icon}" style="font-size: 3rem; color: #dc2626;"></i>
            </div>
            <h5 class="modal-title mb-3" style="font-weight: 700; color: #1f2937;">${title}</h5>
            <div class="error-modal-message" style="color: #6b7280; font-size: 0.95rem;">
              ${message}
            </div>
            <div class="mt-4">
              <button type="button" class="btn btn-primary" data-bs-dismiss="modal" style="border-radius: 10px; padding: 8px 24px;">
                <i class="fas fa-check me-2"></i>Understood
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  `;
  
  // Add modal to body
  document.body.insertAdjacentHTML('beforeend', modalHTML);
  
  // Show modal
  const modal = new bootstrap.Modal(document.getElementById('creditErrorModal'));
  modal.show();
  
  // Clean up modal after it's hidden
  modal._element.addEventListener('hidden.bs.modal', function() {
    document.getElementById('creditErrorModal').remove();
  });
}

// Function to show credit confirmation modal
function showCreditConfirmModal(title, message, icon, alertClass, onConfirm, onCancel) {
  // Remove existing modal if any
  const existingModal = document.getElementById('creditConfirmModal');
  if (existingModal) {
    existingModal.remove();
  }
  
  // Determine colors based on alert class
  let iconColor = '#3b82f6';
  let confirmBtnClass = 'btn-primary';
  let headerClass = 'bg-primary';
  
  if (alertClass === 'danger') {
    iconColor = '#dc2626';
    confirmBtnClass = 'btn-danger';
    headerClass = 'bg-danger';
  } else if (alertClass === 'warning') {
    iconColor = '#f59e0b';
    confirmBtnClass = 'btn-warning';
    headerClass = 'bg-warning';
  } else if (alertClass === 'info') {
    iconColor = '#06b6d4';
    confirmBtnClass = 'btn-info';
    headerClass = 'bg-info';
  }
  
  // Create modal HTML
  const modalHTML = `
    <div class="modal fade" id="creditConfirmModal" tabindex="-1" aria-labelledby="creditConfirmModalLabel" aria-hidden="true">
      <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content" style="border: none; border-radius: 15px; box-shadow: 0 10px 30px rgba(0,0,0,0.2);">
          <div class="modal-header ${headerClass} text-white" style="border-radius: 15px 15px 0 0; border: none;">
            <h5 class="modal-title d-flex align-items-center" id="creditConfirmModalLabel">
              <i class="${icon} me-2"></i>${title}
            </h5>
          </div>
          <div class="modal-body p-4">
            <div class="confirm-modal-message" style="color: #374151; font-size: 0.95rem;">
              ${message}
            </div>
            <div class="mt-4 text-center">
              <strong>Do you want to proceed with adding these credit items?</strong>
            </div>
          </div>
          <div class="modal-footer" style="border: none; padding: 1rem 1.5rem 1.5rem;">
            <button type="button" class="btn btn-secondary" id="creditCancelBtn" style="border-radius: 10px; padding: 8px 20px;">
              <i class="fas fa-times me-2"></i>Cancel
            </button>
            <button type="button" class="btn ${confirmBtnClass}" id="creditConfirmBtn" style="border-radius: 10px; padding: 8px 20px;">
              <i class="fas fa-check me-2"></i>Proceed
            </button>
          </div>
        </div>
      </div>
    </div>
  `;
  
  // Add modal to body
  document.body.insertAdjacentHTML('beforeend', modalHTML);
  
  // Show modal
  const modal = new bootstrap.Modal(document.getElementById('creditConfirmModal'));
  modal.show();
  
  // Add event listeners
  document.getElementById('creditConfirmBtn').addEventListener('click', function() {
    modal.hide();
    if (onConfirm) onConfirm();
  });
  
  document.getElementById('creditCancelBtn').addEventListener('click', function() {
    modal.hide();
    if (onCancel) onCancel();
  });
  
  // Clean up modal after it's hidden
  modal._element.addEventListener('hidden.bs.modal', function() {
    document.getElementById('creditConfirmModal').remove();
  });
}

// Function to show detailed success modal
function showDetailedSuccessModal(mainMessage, messageStatus, failedStatus) {
  // Remove existing modal if any
  const existingModal = document.getElementById('detailedSuccessModal');
  if (existingModal) {
    existingModal.remove();
  }
  
  // Format message status - split by periods and create list items
  let messageStatusHTML = '';
  if (messageStatus) {
    const messages = messageStatus.split('.').filter(msg => msg.trim());
    messageStatusHTML = `
      <div class="alert alert-info">
        <h6><i class="fas fa-paper-plane me-2"></i>Messages Sent Successfully:</h6>
        <ul class="mb-0">
          ${messages.map(msg => `<li>${msg.trim()}</li>`).join('')}
        </ul>
      </div>
    `;
  }
  
  // Format failed status - split by commas and create list items
  let failedStatusHTML = '';
  if (failedStatus) {
    const failedMessages = failedStatus.replace('Failed:', '').split(',').filter(msg => msg.trim());
    failedStatusHTML = `
      <div class="alert alert-warning">
        <h6><i class="fas fa-exclamation-triangle me-2"></i>Messages Not Sent:</h6>
        <ul class="mb-0">
          ${failedMessages.map(msg => `<li>${msg.trim()}</li>`).join('')}
        </ul>
      </div>
    `;
  }
  
  // Create modal HTML
  const modalHTML = `
    <div class="modal fade" id="detailedSuccessModal" tabindex="-1" aria-labelledby="detailedSuccessModalLabel" aria-hidden="true">
      <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
          <div class="modal-header bg-success text-white">
            <h5 class="modal-title" id="detailedSuccessModalLabel">
              <i class="fas fa-check-circle me-2"></i>Success!
            </h5>
            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="alert alert-success">
              <strong>${mainMessage}</strong>
            </div>
            
            ${messageStatusHTML}
            
            ${failedStatusHTML}
            
            ${!messageStatus && !failedStatus ? `
            <div class="alert alert-info">
              <h6><i class="fas fa-info-circle me-2"></i>No Notifications Sent:</h6>
              <p class="mb-0">No reminder types were selected or configured.</p>
            </div>
            ` : ''}
            
            <div class="text-center mt-3">
              <button type="button" class="btn btn-success" data-bs-dismiss="modal">
                <i class="fas fa-check me-2"></i>OK
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  `;
  
  // Add modal to body
  document.body.insertAdjacentHTML('beforeend', modalHTML);
  
  // Show modal
  const modal = new bootstrap.Modal(document.getElementById('detailedSuccessModal'));
  modal.show();
  
  // Auto-close after 5 seconds
  setTimeout(() => {
    const modalInstance = bootstrap.Modal.getInstance(document.getElementById('detailedSuccessModal'));
    if (modalInstance) {
      modalInstance.hide();
    }
  }, 5000);
}
</script>

<!-- Initialize print buttons when document is ready -->
<script>
// Add a print lock variable
let isPrinting = false;

document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, initializing print buttons...'); // Debug log
    var printButtons = document.querySelectorAll('.print-credit-btn');
    console.log('Found print buttons:', printButtons.length); // Debug log
    
    printButtons.forEach(function(btn) {
        console.log('Adding click listener to button for customer:', btn.getAttribute('data-customer-id'));
    btn.addEventListener('click', function(e) {
        e.preventDefault();
            console.log('Print button clicked for customer: ' + this.getAttribute('data-customer-id'));
        
        if (isPrinting) {
            console.log('Print already in progress, please wait...');
            return;
        }
        
        try {
            isPrinting = true;
            var cid = this.getAttribute('data-customer-id');
            console.log('Print button clicked for customer:', cid);
            
            var template = document.getElementById('unpaid-receipt-template-' + cid);
            if (!template) {
                    throw new Error('Receipt template not found for customer: ' + cid);
            }

            // Extract only the <tr> rows from the template's table (skip any non-<tr> content)
            var tempDiv = document.createElement('div');
            tempDiv.innerHTML = template.innerHTML;
            var trRows = '';
            var table = tempDiv.querySelector('table');
            if (table) {
                var trs = table.querySelectorAll('tbody tr');
                var currentDate = '';
                var currentInvoice = '';
                var currentSection = '';
                
                trs.forEach(function(tr) {
                    var cells = tr.querySelectorAll('td');
                    if (cells.length >= 9) {
                        var date = cells[0].textContent.trim();
                        var invoice = cells[1].textContent.trim();
                        
                        // Start new section if date or invoice changes
                        if (date !== currentDate || invoice !== currentInvoice) {
                            if (currentSection !== '') {
                                trRows += currentSection + '</table></div>';
                            }
                            currentDate = date;
                            currentInvoice = invoice;
                            currentSection = '<div style="margin-bottom: 15px;"><div style="font-weight: bold; margin-bottom: 5px;">Date: ' + date + ' | Invoice: ' + invoice + '</div><table><thead><tr><th>Items</th><th>Qty</th><th>Payment Details</th><th class="text-right">Waived</th><th class="text-right">Credit Amount</th><th class="text-right">Paid</th><th class="text-right">Balance</th></tr></thead><tbody>';
                        }
                        
                        // Add the row without date and invoice columns
                        currentSection += '<tr><td>' + cells[2].textContent + '</td><td>' + cells[3].textContent + '</td><td>' + cells[4].textContent + '</td><td class="text-right">' + cells[5].textContent + '</td><td class="text-right">' + cells[6].textContent + '</td><td class="text-right">' + cells[7].textContent + '</td><td class="text-right">' + cells[8].textContent + '</td></tr>';
                    }
                });
                
                // Close the last section
                if (currentSection !== '') {
                    trRows += currentSection + '</table></div>';
                }
            }

            var printWindow = window.open('', '_blank', 'width=400,height=600');
            if (!printWindow) {
                throw new Error('Pop-up blocked. Please allow pop-ups for this site.');
            }

            // Write the receipt content with only one set of store info and receipt info
            printWindow.document.write(`
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Unpaid Credit Receipt</title>
                    <meta charset="UTF-8">
                    <style>
                        @media print {
                            @page {
                                size: 80mm auto;
                                margin: 0;
                            }
                            body {
                                width: 80mm;
                                margin: 0;
                                padding: 5px;
                            }
                            .no-print {
                                display: none !important;
                            }
                        }
                        body { 
                            font-family: 'Arial Black', Arial, sans-serif;
                            font-size: 12px;
                            color: #000;
                            margin: 0;
                            padding: 5px;
                            width: 80mm;
                            max-width: 80mm;
                            font-weight: 900;
                        }
                        .store-info {
                            text-align: center;
                            margin-bottom: 8px;
                        }
                        .store-name {
                            font-size: 1.2rem;
                            font-weight: 700;
                            margin-bottom: 2px;
                        }
                        .divider {
                            border-top: 1px dashed #000;
                            margin: 8px 0;
                        }
                        .section-title {
                            font-weight: bold;
                            margin: 8px 0;
                            text-align: center;
                        }
                        table {
                            width: 100%;
                            border-collapse: collapse;
                            font-size: 11px;
                            margin-bottom: 10px;
                        }
                        th, td {
                            padding: 2px;
                            text-align: left;
                            border-bottom: 1px solid #ddd;
                        }
                        th {
                            border-bottom: 1px solid #000;
                        }
                        .text-right {
                            text-align: right;
                        }
                        .qr-code {
                            text-align: center;
                            margin: 10px 0;
                        }
                        .qr-code img {
                            width: 120px;
                            height: 120px;
                        }
                        .footer {
                            text-align: center;
                            margin-top: 10px;
                            font-size: 11px;
                        }
                    </style>
                </head>
                <body>
                    <div class="store-info">
                        <div style="font-size:1.1rem; font-weight:900; color:#000; margin-bottom:2px;">UNPAID CREDIT ITEMS <br> RECEIPT</div>
                        <?php if ($logoData): ?>
                            <?php
                            $mimeType = 'image/jpeg'; // Default to JPEG
                            if (strpos(strtolower($logoName), '.png') !== false) {
                                $mimeType = 'image/png';
                            } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                                $mimeType = 'image/gif';
                            }
                            ?>
                            <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
                        <?php endif; ?>
                        <div class="store-name"><?php echo htmlspecialchars($storeName); ?></div>
                        <div><?php echo htmlspecialchars($storeAddress); ?></div>
                        <div>Phone: <?php echo htmlspecialchars($storePhone); ?></div>
                        <div>Email: <?php echo htmlspecialchars($storeEmail); ?></div>
                        <div>Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
                    </div>
                    <div class="divider"></div>
                    <div class="section-title">UNPAID CREDIT ITEMS</div>
                    <div class="divider"></div>
                    <div><b>Customer:</b> <span class="customer-name"></span></div>
                    <div><b>Date:</b> <span class="receipt-date"></span></div>
                    <div><b>Receipt No:</b> <span class="receipt-number"></span></div>
                    <div class="divider"></div>
                    <div id="invoice-sections">
                        ${trRows}
                    </div>
                    <div class="divider"></div>
                    <div><b>Total Balance:</b> <span class="total-balance"></span></div>
                    <div class="divider"></div>
                    <div style="text-align:center; margin:12px 0;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
                    <div class="qr-code">
                        <img id="receipt-qr" src="" alt="Receipt QR Code">
                    </div>
                    <div class="divider"></div>
                    <div class="footer">Designed and developed by Vicotechnologies.co.ke <br>+254702214793</div>
                </body>
                </html>
            `);

            printWindow.document.close();

            printWindow.onload = function() {
                try {
                    var now = new Date();
                    var dateStr = now.getFullYear() + '-' + String(now.getMonth()+1).padStart(2,'0') + '-' + String(now.getDate()).padStart(2,'0') + ' ' + String(now.getHours()).padStart(2,'0') + ':' + String(now.getMinutes()).padStart(2,'0') + ':' + String(now.getSeconds()).padStart(2,'0');
                    var receiptNo = 'UNPAID-' + cid + '-' + now.getTime();
                    
                    // Update receipt elements
                    printWindow.document.querySelector('.customer-name').textContent = template.getAttribute('data-customer-name');
                    printWindow.document.querySelector('.receipt-date').textContent = dateStr;
                    printWindow.document.querySelector('.receipt-number').textContent = receiptNo;
                    printWindow.document.querySelector('.total-balance').textContent = template.getAttribute('data-total-balance');

                    // Generate QR code data (simplified for better scanning)
                    var receiptInfo = {
                        type: 'UNPAID',
                        store: '<?php echo addslashes($storeName); ?>',
                        store_phone: '<?php echo addslashes($storePhone); ?>',
                        store_email: '<?php echo addslashes($storeEmail); ?>',
                        customer: template.getAttribute('data-customer-name'),
                        total_outstanding: template.getAttribute('data-total-balance'),
                        total_paid: template.getAttribute('data-total-paid') || '0',
                        total_waived: template.getAttribute('data-total-waived') || '0',
                        date: dateStr.split(' ')[0],
                        receipt: receiptNo
                    };



                    // Generate QR code using qrserver
                    var qrImg = printWindow.document.getElementById('receipt-qr');
                    qrImg.src = 'https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=' + encodeURIComponent(JSON.stringify(receiptInfo));

                    // Wait for QR code to load before printing
                    qrImg.onload = function() {
                        setTimeout(function() {
                            printWindow.print();
                            setTimeout(function() {
                                printWindow.close();
                                isPrinting = false;
                            }, 1000);
                        }, 500);
                    };

                    // Handle QR code load error
                    qrImg.onerror = function() {
                        console.error('QR code failed to load');
                        setTimeout(function() {
                            printWindow.print();
                            setTimeout(function() {
                                printWindow.close();
                                isPrinting = false;
                            }, 1000);
                        }, 500);
                    };

                } catch (error) {
                    console.error('Error preparing receipt:', error);
                    alert('Error preparing receipt. Please try again.');
                    isPrinting = false;
                }
            };

        } catch (error) {
            console.error('Print button error:', error);
            alert(error.message || 'Error printing receipt. Please try again.');
            isPrinting = false;
        }
    });
    });
});
</script>

<!-- Success Modal -->
<div class="modal fade" id="successModal" tabindex="-1" aria-labelledby="successModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header bg-success text-white border-0">
                <h5 class="modal-title d-flex align-items-center" id="successModalLabel">
                    <svg class="me-2" width="20" height="20" fill="currentColor" viewBox="0 0 16 16">
                        <path d="M16 8A8 8 0 1 1 0 8a8 8 0 0 1 16 0zm-3.97-3.03a.75.75 0 0 0-1.08.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-.01-1.05z"/>
                    </svg>
                    Success
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body text-center py-4" id="successModalBody">
                <!-- Success message will be injected here -->
            </div>
            <div class="modal-footer border-0 justify-content-center">
                <div class="progress w-100" style="height: 3px;">
                    <div class="progress-bar bg-success" id="successProgress" role="progressbar" style="width: 100%"></div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
#successModal .modal-content {
    border-radius: 15px;
    overflow: hidden;
}

#successModal .modal-header {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
}

#successModal .modal-body {
    font-size: 1.1rem;
    font-weight: 500;
    color: #495057;
}

#successModal .progress {
    border-radius: 0;
    background-color: #e9ecef;
}

#successModal .progress-bar {
    transition: width 3s linear;
}

#successModal .btn-close-white {
    filter: brightness(0) invert(1);
}
</style>

<script>
// Function to show success modal with auto-dismiss
function showSuccessModal(message) {
    const successModal = new bootstrap.Modal(document.getElementById('successModal'));
    document.getElementById('successModalBody').textContent = message;
    
    // Reset progress bar
    const progressBar = document.getElementById('successProgress');
    progressBar.style.width = '100%';
    
    successModal.show();
    
    // Start progress bar animation
    setTimeout(() => {
        progressBar.style.width = '0%';
    }, 100);
    
    // Auto-dismiss after 3 seconds
    setTimeout(() => {
        successModal.hide();
    }, 3000);
}

// Initialize print buttons when document is ready
</script>

<?php 
echo getUniversalPermissionModalHTML();
echo getUniversalPermissionModalJS();
?>

<?php include '../includes/footer.php'; ?>
</body>
</html> 
_rewritten_file>
